<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use Illuminate\Http\Request;

class BankController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $banks = Bank::orderBy('bank_name')->paginate(20);
        return view('admin.banks.index', compact('banks'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.banks.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'bank_name' => 'required|string|max:255|unique:banks,bank_name',
            'bank_code' => 'nullable|string|max:50|unique:banks,bank_code',
            'swift_code' => 'nullable|string|max:50',
            'address' => 'nullable|string',
            'contact_number' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        Bank::create($validated);

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank added successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Bank $bank)
    {
        return view('admin.banks.show', compact('bank'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Bank $bank)
    {
        return view('admin.banks.edit', compact('bank'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Bank $bank)
    {
        $validated = $request->validate([
            'bank_name' => 'required|string|max:255|unique:banks,bank_name,' . $bank->id,
            'bank_code' => 'nullable|string|max:50|unique:banks,bank_code,' . $bank->id,
            'swift_code' => 'nullable|string|max:50',
            'address' => 'nullable|string',
            'contact_number' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        $bank->update($validated);

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Bank $bank)
    {
        $bank->delete();

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank deleted successfully!');
    }

    /**
     * Get active banks for dropdown
     */
    public function getActiveBanks()
    {
        return Bank::active()->orderBy('bank_name')->get(['id', 'bank_name']);
    }
}
