<?php

namespace App\Http\Controllers\Admin\Finance\Banking;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\BankAccount;
use App\Models\BankReconciliation;
use App\Models\BankReconciliationLine;
use App\Models\Cheque;
use App\Models\BankDiscrepancy;

class BankingController extends Controller
{
    public function accounts()
    {
        $accounts = BankAccount::latest()->paginate(15);
        return view('admin.finance.banking.accounts', compact('accounts'));
    }

    public function storeAccount(Request $request)
    {
        $request->validate([
            'bank_name' => 'required|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_number' => 'required|string|unique:bank_accounts,account_number',
            'branch' => 'required|string|max:255',
            'ifsc_code' => 'required|string|max:255',
        ]);

        BankAccount::create($request->all());

        return redirect()->route('admin.finance.banking.accounts')->with('success', 'Bank account added successfully.');
    }

    public function updateAccount(Request $request, BankAccount $bankAccount)
    {
        $request->validate([
            'bank_name' => 'required|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_number' => 'required|string|unique:bank_accounts,account_number,' . $bankAccount->id,
            'branch' => 'required|string|max:255',
            'ifsc_code' => 'required|string|max:255',
        ]);

        $bankAccount->update($request->all());

        return redirect()->route('admin.finance.banking.accounts')->with('success', 'Bank account updated successfully.');
    }

    public function destroyAccount(BankAccount $bankAccount)
    {
        $bankAccount->delete();

        return redirect()->route('admin.finance.banking.accounts')->with('success', 'Bank account deleted successfully.');
    }

    public function reconciliation()
    {
        $recons = BankReconciliation::latest()->paginate(15);
        return view('admin.finance.banking.reconciliation', compact('recons'));
    }

    public function storeReconciliation(Request $request)
    {
        $request->validate([
            'bank_name' => 'required|string|max:255',
            'period' => 'required|string',
            'statement' => 'required|file|mimes:csv,txt',
        ]);

        $rec = BankReconciliation::create([
            'bank_name' => $request->bank_name,
            'statement_period' => $request->period,
            'uploaded_at' => now(),
        ]);

        // Simple CSV parser: expects headers date,description,amount (case-insensitive)
        if (($handle = fopen($request->file('statement')->getRealPath(), 'r')) !== false) {
            $headers = fgetcsv($handle);
            $map = [];
            if ($headers) {
                foreach ($headers as $idx => $h) {
                    $key = strtolower(trim($h));
                    $map[$key] = $idx;
                }
            }
            while (($row = fgetcsv($handle)) !== false) {
                $date = $map['date'] ?? $map['txn_date'] ?? null;
                $desc = $map['description'] ?? $map['details'] ?? null;
                $amt = $map['amount'] ?? $map['amt'] ?? null;
                if ($date === null || $desc === null || $amt === null) {
                    continue;
                }
                $txnDate = @date('Y-m-d', strtotime($row[$date] ?? '')) ?: null;
                $description = (string)($row[$desc] ?? '');
                $amount = is_numeric($row[$amt] ?? null) ? (float)$row[$amt] : null;
                if ($txnDate && $description !== '' && $amount !== null) {
                    BankReconciliationLine::create([
                        'bank_reconciliation_id' => $rec->id,
                        'txn_date' => $txnDate,
                        'description' => $description,
                        'amount' => $amount,
                        'is_matched' => false,
                    ]);
                }
            }
            fclose($handle);
        }

        return redirect()->route('admin.finance.banking.reconciliation.show', $rec)->with('success', 'Statement imported successfully.');
    }

    public function showReconciliation(BankReconciliation $rec, Request $request)
    {
        $linesQuery = $rec->lines()->orderByDesc('txn_date');
        if ($request->filled('status')) {
            if ($request->status === 'matched') $linesQuery->where('is_matched', true);
            if ($request->status === 'unmatched') $linesQuery->where('is_matched', false);
        }
        $lines = $linesQuery->paginate(50);
        return view('admin.finance.banking.reconciliation-show', compact('rec','lines'));
    }

    public function matchLine(BankReconciliationLine $line)
    {
        $line->update(['is_matched' => !$line->is_matched]);
        return back();
    }

    public function cheques(Request $request)
    {
        $accounts = BankAccount::orderBy('bank_name')->get();
        $q = Cheque::with('bankAccount')->latest();
        if ($request->filled('status')) $q->where('status', $request->status);
        if ($request->filled('type')) $q->where('type', $request->type);
        if ($request->filled('bank_account_id')) $q->where('bank_account_id', $request->bank_account_id);
        $cheques = $q->paginate(15);
        return view('admin.finance.banking.cheques', compact('cheques','accounts'));
    }

    public function storeCheque(Request $request)
    {
        $validated = $request->validate([
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'cheque_number' => 'required|string|unique:cheques,cheque_number',
            'type' => 'required|in:incoming,outgoing',
            'payee' => 'nullable|string',
            'amount' => 'required|numeric|min:0',
            'issue_date' => 'required|date',
            'due_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);
        Cheque::create($validated);
        return redirect()->route('admin.finance.banking.cheques')->with('success','Cheque recorded successfully');
    }

    public function updateCheque(Request $request, Cheque $cheque)
    {
        $validated = $request->validate([
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'cheque_number' => 'required|string|unique:cheques,cheque_number,'.$cheque->id,
            'type' => 'required|in:incoming,outgoing',
            'payee' => 'nullable|string',
            'amount' => 'required|numeric|min:0',
            'issue_date' => 'required|date',
            'due_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'status' => 'nullable|string'
        ]);
        $cheque->update($validated);
        return back()->with('success','Cheque updated');
    }

    public function destroyCheque(Cheque $cheque)
    {
        $cheque->delete();
        return back()->with('success','Cheque deleted');
    }

    public function changeChequeStatus(Request $request, Cheque $cheque)
    {
        $request->validate(['status' => 'required|in:issued,deposited,cleared,bounced,cancelled']);
        $data = ['status' => $request->status];
        if ($request->status === 'deposited') $data['deposit_date'] = now();
        if ($request->status === 'cleared') $data['cleared_at'] = now();
        if (in_array($request->status, ['bounced','cancelled','issued'])) {
            $data['cleared_at'] = null;
        }
        $cheque->update($data);
        return back()->with('success','Status updated');
    }

    public function discrepancies(Request $request)
    {
        $q = BankDiscrepancy::with('line')->latest();
        if ($request->filled('status')) {
            $q->where('resolved', $request->status==='resolved');
        }
        $alerts = $q->paginate(20);
        return view('admin.finance.banking.discrepancies', compact('alerts'));
    }

    public function storeDiscrepancy(Request $request)
    {
        $validated = $request->validate([
            'bank_reconciliation_line_id' => 'nullable|exists:bank_reconciliation_lines,id',
            'source' => 'required|in:bank,ledger,manual',
            'txn_date' => 'nullable|date',
            'description' => 'required|string',
            'amount' => 'required|numeric',
        ]);
        BankDiscrepancy::create($validated);
        return back()->with('success','Discrepancy created');
    }

    public function resolveDiscrepancy(Request $request, BankDiscrepancy $discrepancy)
    {
        $discrepancy->update([
            'resolved' => true,
            'resolved_at' => now(),
            'resolution_notes' => $request->input('resolution_notes'),
        ]);
        return back()->with('success','Discrepancy resolved');
    }
}
