<?php

namespace App\Http\Controllers\Admin\Finance;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\BankDeposit;
use App\Models\CodCollection;
use App\Models\CodPendingReport;
use App\Models\CodVarianceReport;
use App\Models\MerchantPayment;
use App\Models\Bank;

class CODController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:finance.view')->only(['dailyCollection', 'showCollection', 'deposits', 'showDeposit', 'pendingReport', 'varianceReport']);
        $this->middleware('can:finance.add')->only(['storeCollection', 'storeDeposit']);
        $this->middleware('can:finance.edit')->only(['editCollection', 'updateCollection', 'editDeposit', 'updateDeposit', 'verifyDeposit', 'resolveVariance']);
    }
    // Daily Collection
    public function dailyCollection(Request $request)
    {
        $date = $request->get('date', date('Y-m-d'));

        // Get collections for the selected date
        $collections = CodCollection::where('collection_date', $date)
            ->with('rider')
            ->latest()
            ->get();

        // Calculate stats for the selected date
        $stats = [
            'today_collection' => CodCollection::where('collection_date', $date)
                ->where('status', 'completed')
                ->sum('collected_amount'),
            'pending' => CodCollection::where('collection_date', $date)
                ->where('status', 'pending')
                ->sum('pending_amount'),
            'collected' => CodCollection::where('collection_date', $date)
                ->where('status', 'completed')
                ->sum('collected_amount'),
            'total_riders' => CodCollection::where('collection_date', $date)
                ->distinct('rider_id')
                ->count('rider_id'),
        ];

        return view('admin.finance.cod.daily-collection', compact('stats', 'date', 'collections'));
    }

    public function storeCollection(Request $request)
    {
        $validated = $request->validate([
            'rider_id' => 'required',
            'amount' => 'required|numeric',
            'parcels' => 'required|integer',
        ]);

        return redirect()->route('admin.finance.cod.daily-collection')
            ->with('success', 'Collection recorded successfully!');
    }

    public function showCollection($id)
    {
        return view('admin.finance.cod.collection-show', ['id' => $id]);
    }

    public function editCollection($id)
    {
        return view('admin.finance.cod.collection-edit', ['id' => $id]);
    }

    public function updateCollection(Request $request, $id)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric',
            'status' => 'required|string',
        ]);

        return redirect()->route('admin.finance.cod.daily-collection')
            ->with('success', 'Collection updated successfully!');
    }

    public function deposits(Request $request)
    {
        $date = $request->get('date', date('Y-m-d'));

        // Get deposits for the selected date
        $deposits = BankDeposit::where('deposit_date', $date)
            ->with('depositedBy')
            ->latest()
            ->get();

        // Calculate stats for the selected date
        $stats = [
            'today_deposits' => BankDeposit::where('deposit_date', $date)
                ->where('status', 'completed')
                ->sum('amount'),
            'pending' => BankDeposit::where('deposit_date', $date)
                ->where('status', 'pending')
                ->sum('amount'),
            'verified' => BankDeposit::where('deposit_date', $date)
                ->where('status', 'completed')
                ->sum('amount'),
            'total_count' => BankDeposit::where('deposit_date', $date)
                ->count(),
        ];

        // Get active banks for dropdown
        $banks = Bank::active()->orderBy('bank_name')->get();

        return view('admin.finance.cod.deposits', compact('stats', 'date', 'deposits', 'banks'));
    }
    public function storeDeposit(Request $request)
    {
        $validated = $request->validate([
            'bank_name' => 'required|string',
            'amount' => 'required|numeric|min:0',
            'reference_number' => 'required|string|unique:bank_deposits,reference_number',
            'deposit_date' => 'required|date',
            'branch_name' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Create the deposit record
        BankDeposit::create([
            'bank_name' => $validated['bank_name'],
            'branch_name' => $validated['branch_name'] ?? null,
            'amount' => $validated['amount'],
            'reference_number' => $validated['reference_number'],
            'deposit_date' => $validated['deposit_date'],
            'notes' => $validated['notes'] ?? null,
            'deposited_by' => auth()->id(),
            'status' => 'pending', // Bank deposits need verification
        ]);

        return redirect()->route('admin.finance.cod.deposits')
            ->with('success', 'Deposit recorded successfully and awaiting verification!');
    }

    public function showDeposit($id)
    {
        $deposit = BankDeposit::with('depositedBy')->findOrFail($id);
        return view('admin.finance.cod.deposit-show', compact('deposit'));
    }

    public function editDeposit($id)
    {
        $deposit = BankDeposit::findOrFail($id);
        return view('admin.finance.cod.deposit-edit', compact('deposit'));
    }

    public function updateDeposit(Request $request, $id)
    {
        $validated = $request->validate([
            'bank_name' => 'required|string',
            'amount' => 'required|numeric|min:0',
            'reference_number' => 'required|string|unique:bank_deposits,reference_number,' . $id,
            'deposit_date' => 'required|date',
            'branch_name' => 'nullable|string',
            'status' => 'required|string',
            'notes' => 'nullable|string',
        ]);

        $deposit = BankDeposit::findOrFail($id);

        $deposit->update([
            'bank_name' => $validated['bank_name'],
            'branch_name' => $validated['branch_name'] ?? null,
            'amount' => $validated['amount'],
            'reference_number' => $validated['reference_number'],
            'deposit_date' => $validated['deposit_date'],
            'status' => $validated['status'],
            'notes' => $validated['notes'] ?? null,
        ]);

        return redirect()->route('admin.finance.cod.deposits')
            ->with('success', 'Deposit updated successfully!');
    }

    public function verifyDeposit(Request $request, $id)
    {
        try {
            $deposit = BankDeposit::findOrFail($id);

            if ($deposit->status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only pending deposits can be verified'
                ], 400);
            }

            $deposit->update([
                'status' => 'completed',
                'verified_by' => auth()->id(),
                'verified_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Deposit verified successfully!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error verifying deposit: ' . $e->getMessage()
            ], 500);
        }
    }

    // Reports
    public function pendingReport(Request $request)
    {
        $date = $request->get('date', date('Y-m-d'));
        
        // Get pending COD for the selected date
        $pendingData = []; // Replace with actual query
        
        // Calculate stats for the selected date
        $stats = [
            'total_pending' => 245000, // Replace with actual calculation
            'pending_parcels' => 156,
            'overdue' => 85000,
            'today_due' => 160000
        ];
        
        return view('admin.finance.cod.pending-report', compact('stats', 'date'));
    }

    public function varianceReport(Request $request)
    {
        $date = $request->get('date', date('Y-m-d'));
        
        // Get variance data for the selected date
        $varianceData = []; // Replace with actual query
        
        // Calculate stats for the selected date
        $stats = [
            'total_variance' => 12500, // Replace with actual calculation
            'cases' => 8,
            'resolved' => 5000,
            'pending' => 7500
        ];
        
        return view('admin.finance.cod.variance-report', compact('stats', 'date'));
    }

    public function resolveVariance(Request $request, $id)
    {
        $validated = $request->validate([
            'resolution' => 'required|string',
            'notes' => 'nullable|string',
        ]);

        return back()->with('success', 'Variance resolved successfully!');
    }

    /**
     * Export daily collection report as CSV
     */
    public function exportDailyCollection(Request $request)
    {
        $date = $request->get('date', date('Y-m-d'));

        $collections = CodCollection::where('collection_date', $date)
            ->with('rider')
            ->get(['rider_id', 'collected_amount', 'pending_amount', 'total_parcels']);

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="daily-collection-'.$date.'.csv"',
        ];

        $callback = function() use ($collections) {
            $out = fopen('php://output', 'w');
            fputcsv($out, ['Rider', 'Collected Amount', 'Pending Amount', 'Parcels']);
            foreach ($collections as $row) {
                fputcsv($out, [
                    optional($row->rider)->name,
                    $row->collected_amount,
                    $row->pending_amount,
                    $row->total_parcels,
                ]);
            }
            fclose($out);
        };

        return response()->stream($callback, 200, $headers);
    }
}
