<?php

namespace App\Http\Controllers\Admin\Finance\Expense;

use App\Http\Controllers\Controller;
use App\Models\Expense;
use App\Models\Vendor;
use Illuminate\Http\Request;

class ExpenseController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:finance.view')->only(['companyExpenses', 'branchExpenses', 'vendorPayments', 'approvalFlow', 'recurring']);
        $this->middleware('can:finance.add')->only(['store']);
        $this->middleware('can:finance.edit')->only(['edit', 'update', 'approve']);
        $this->middleware('can:finance.delete')->only(['destroy']);
    }
    public function companyExpenses(Request $request)
    {
        $query = Expense::with('vendor');
        
        if ($request->filled('start_date')) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }
        if ($request->filled('expense_category')) {
            $query->where('expense_category', $request->expense_category);
        }
        
        $expenses = $query->latest()->paginate(15);
        return view('admin.finance.expense.company', compact('expenses'));
    }

    public function branchExpenses(Request $request)
    {
        $query = Expense::with('vendor');
        if ($request->filled('start_date')) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }
        $expenses = $query->latest()->paginate(15);
        return view('admin.finance.expense.branch', compact('expenses'));
    }

    public function vendorPayments(Request $request)
    {
        $query = Expense::with('vendor')->whereNotNull('vendor_id');
        if ($request->filled('start_date')) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }
        $expenses = $query->latest()->paginate(15);
        return view('admin.finance.expense.vendorPayments', compact('expenses'));
    }

    public function approvalFlow()
    {
        $pending = Expense::where('status', 'pending')->latest()->paginate(15);
        $approved = Expense::where('status', 'approved')->latest()->paginate(15);
        return view('admin.finance.expense.approvalFlow', compact('pending', 'approved'));
    }

    public function recurring(Request $request)
    {
        // Placeholder: filter expenses with recurring flag if available
        $expenses = Expense::latest()->paginate(15);
        return view('admin.finance.expense.recurring', compact('expenses'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'expense_category' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'expense_date' => 'required|date',
            'vendor_id' => 'nullable|exists:vendors,id',
            'receipt' => 'nullable|file|mimes:jpg,png,pdf|max:2048',
        ]);

        $data = $request->all();

        // Generate expense_number like EXP-YYYYMMDD-XXXX
        $data['expense_number'] = 'EXP-' . date('Ymd') . '-' . str_pad((string) random_int(1, 9999), 4, '0', STR_PAD_LEFT);

        if ($request->hasFile('receipt')) {
            $data['receipt'] = $request->file('receipt')->store('receipts', 'public');
        }

        Expense::create($data);

        return redirect()->route('finance.expenses.company')->with('success', 'Expense added successfully.');
    }

    public function edit(Expense $expense)
    {
        $vendors = Vendor::all();
        return view('admin.finance.expense.edit', compact('expense', 'vendors'));
    }

    public function update(Request $request, Expense $expense)
    {
        $request->validate([
            'expense_category' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'expense_date' => 'required|date',
            'vendor_id' => 'nullable|exists:vendors,id',
            'receipt' => 'nullable|file|mimes:jpg,png,pdf|max:2048',
        ]);

        $data = $request->all();

        if ($request->hasFile('receipt')) {
            $data['receipt'] = $request->file('receipt')->store('receipts', 'public');
        }

        $expense->update($data);

        return redirect()->route('finance.expenses.company')->with('success', 'Expense updated successfully.');
    }

    public function destroy(Expense $expense)
    {
        $expense->delete();

        return redirect()->route('finance.expenses.company')->with('success', 'Expense deleted successfully.');
    }

    public function approve(Expense $expense)
    {
        if ($expense->status !== 'approved') {
            $expense->update(['status' => 'approved']);
        }
        return back()->with('success', 'Expense approved successfully.');
    }
}
