<?php

namespace App\Http\Controllers\Admin\Finance\Settlement;

use App\Http\Controllers\Controller;
use App\Models\MerchantInvoice;
use App\Models\InvoiceItem;
use App\Models\User;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:finance.view')->only(['index', 'show']);
        $this->middleware('can:finance.add')->only(['create', 'store']);
        $this->middleware('can:finance.edit')->only(['edit', 'update', 'send', 'markAsPaid']);
        $this->middleware('can:finance.delete')->only(['destroy']);
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = MerchantInvoice::with('merchant');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('merchant_id')) {
            $query->where('merchant_id', $request->merchant_id);
        }

        if ($request->filled('start_date')) {
            $query->whereDate('invoice_date', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $query->whereDate('invoice_date', '<=', $request->end_date);
        }

        $invoices = $query->latest()->paginate(15);
        $merchants = \App\Models\User::where('role', 'seller')->get();

        return view('admin.finance.settlements.invoices.index', compact('invoices', 'merchants'));
    }

    /**
     */
    public function create()
    {
        // Fetch all merchants from database
        $merchants = User::where('role', 'seller')
            ->orderBy('name')
            ->get(['id', 'name', 'email', 'company_name']);
        
        // Get statistics for context
        $stats = [
            'total_merchants' => User::where('role', 'seller')->count(),
            'pending_invoices' => MerchantInvoice::where('status', 'pending')->count(),
            'draft_invoices' => MerchantInvoice::where('status', 'draft')->count(),
        ];
        
        return view('admin.finance.settlements.invoices.create', compact('merchants', 'stats'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'merchant_id' => 'required|exists:users,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'subtotal' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        $total = $validated['subtotal'] + ($validated['tax_amount'] ?? 0) - ($validated['discount_amount'] ?? 0);

        $invoice = MerchantInvoice::create([
            'merchant_id' => $validated['merchant_id'],
            'invoice_date' => $validated['invoice_date'],
            'due_date' => $validated['due_date'],
            'subtotal' => $validated['subtotal'],
            'tax_amount' => $validated['tax_amount'] ?? 0,
            'discount_amount' => $validated['discount_amount'] ?? 0,
            'total_amount' => $total,
            'notes' => $validated['notes'] ?? null,
            'status' => 'draft',
        ]);

        // Create invoice items
        foreach ($validated['items'] as $item) {
            InvoiceItem::create([
                'invoice_id' => $invoice->id,
                'description' => $item['description'],
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'total_price' => $item['quantity'] * $item['unit_price'],
            ]);
        }

        return redirect()->route('admin.finance.settlements.invoices.index')
            ->with('success', 'Invoice created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(MerchantInvoice $invoice)
    {
        $invoice->load('merchant', 'items');
        return view('admin.finance.settlements.invoices.show', compact('invoice'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MerchantInvoice $invoice)
    {
        $invoice->load('items');
        $merchants = User::where('role', 'seller')->get();
        return view('admin.finance.settlements.invoices.edit', compact('invoice', 'merchants'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MerchantInvoice $invoice)
    {
        $validated = $request->validate([
            'merchant_id' => 'required|exists:users,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'subtotal' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'status' => 'required|in:draft,sent,paid,overdue,cancelled',
            'notes' => 'nullable|string',
        ]);

        $total = $validated['subtotal'] + ($validated['tax_amount'] ?? 0) - ($validated['discount_amount'] ?? 0);

        $invoice->update([
            'merchant_id' => $validated['merchant_id'],
            'invoice_date' => $validated['invoice_date'],
            'due_date' => $validated['due_date'],
            'subtotal' => $validated['subtotal'],
            'tax_amount' => $validated['tax_amount'] ?? 0,
            'discount_amount' => $validated['discount_amount'] ?? 0,
            'total_amount' => $total,
            'status' => $validated['status'],
            'notes' => $validated['notes'] ?? null,
        ]);

        return redirect()->route('admin.finance.settlements.invoices.index')
            ->with('success', 'Invoice updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MerchantInvoice $invoice)
    {
        $invoice->items()->delete();
        $invoice->delete();

        return redirect()->route('admin.finance.settlements.invoices.index')
            ->with('success', 'Invoice deleted successfully!');
    }

    /**
     * Send invoice to merchant
     */
    public function send(MerchantInvoice $invoice)
    {
        $invoice->update(['status' => 'sent']);

        return back()->with('success', 'Invoice sent successfully!');
    }

    /**
     * Mark invoice as paid
     */
    public function markAsPaid(MerchantInvoice $invoice)
    {
        $invoice->update(['status' => 'paid']);

        return back()->with('success', 'Invoice marked as paid!');
    }
}
