<?php

namespace App\Http\Controllers\Admin\Finance\Taxation;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\TaxRate;

class TaxationController extends Controller
{
    public function index()
    {
        return view('admin.finance.taxation.index');
    }

    public function gstCalculation()
    {
        $taxRates = TaxRate::where('type', 'GST')->latest()->paginate(15);
        return view('admin.finance.taxation.gst-calculation', compact('taxRates'));
    }

    public function storeTaxRate(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric|min:0',
            'type' => 'required|string|max:255',
        ]);

        TaxRate::create($request->all());

        return redirect()->back()->with('success', 'Tax rate created successfully.');
    }

    public function updateTaxRate(Request $request, TaxRate $taxRate)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric|min:0',
            'type' => 'required|string|max:255',
        ]);

        $taxRate->update($request->all());

        return redirect()->back()->with('success', 'Tax rate updated successfully.');
    }

    public function destroyTaxRate(TaxRate $taxRate)
    {
        $taxRate->delete();

        return redirect()->back()->with('success', 'Tax rate deleted successfully.');
    }

    public function incomeTax(Request $request)
    {
        $query = \App\Models\IncomeTaxWithholding::with('employee');

        if ($request->filled('period')) {
            $query->where('period', $request->period);
        }

        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }

        $withholdings = $query->latest()->paginate(20);
        $employees = \App\Models\Employee::orderBy('name')->get();

        return view('admin.finance.taxation.income-tax', compact('withholdings', 'employees'));
    }

    public function storeIncomeTax(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'period' => 'required|date_format:Y-m',
            'gross_income' => 'required|numeric|min:0',
            'taxable_income' => 'required|numeric|min:0',
            'tax_withheld' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        \App\Models\IncomeTaxWithholding::create($validated);

        return redirect()->route('admin.finance.taxation.income-tax')->with('success', 'Income tax withholding record created successfully.');
    }

    public function filings(Request $request)
    {
        $query = \App\Models\TaxFiling::with('filedBy');

        if ($request->filled('period')) {
            $query->where('period', $request->period);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $filings = $query->latest()->paginate(20);

        return view('admin.finance.taxation.filings', compact('filings'));
    }

    public function eFiling()
    {
        return view('admin.finance.taxation.e-filing');
    }

    public function deductionReports(Request $request)
    {
        $query = \App\Models\TaxDeduction::with('employee');

        if ($request->filled('month')) {
            $query->where('month', $request->month);
        }

        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }

        $deductions = $query->latest()->paginate(20);
        $employees = \App\Models\Employee::orderBy('name')->get();

        return view('admin.finance.taxation.deduction-reports', compact('deductions', 'employees'));
    }

    public function calendarAlerts()
    {
        $alerts = \App\Models\TaxAlert::latest()->get();
        return view('admin.finance.taxation.calendar-alerts', compact('alerts'));
    }

    public function storeCalendarAlert(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'alert_date' => 'required|date',
            'frequency' => 'nullable|string|in:,monthly,quarterly,annually',
            'description' => 'nullable|string',
        ]);

        // Default active true
        $validated['is_active'] = true;

        \App\Models\TaxAlert::create($validated);

        return redirect()->route('admin.finance.taxation.calendar-alerts')
            ->with('success', 'Tax alert created successfully.');
    }

    public function editTaxRate(TaxRate $taxRate)
    {
        return view('admin.finance.taxation.edit-tax-rate', compact('taxRate'));
    }

    public function submitFiling(Request $request)
    {
        // Submit filing logic
        return redirect()->back()->with('success', 'Filing submitted successfully');
    }
}
