<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\MerchantInvoice;
use App\Models\InvoiceItem;
use App\Models\MerchantPayment;
use App\Models\Expense;
use App\Models\CompanyAccount;
use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class FinanceController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:finance.view')->only(['invoices', 'showInvoice']);
        $this->middleware('can:finance.add')->only(['createInvoice', 'storeInvoice']);
        $this->middleware('can:finance.edit')->only(['editInvoice', 'updateInvoice']);
        $this->middleware('can:finance.delete')->only(['destroyInvoice']);
    }
    public function dashboard()
    {
        $stats = [
            'total_merchants' => User::where('role', 'seller')->count(),
            'total_invoices' => MerchantInvoice::count(),
            'pending_payments' => MerchantPayment::where('status', 'pending')->count(),
            'total_expenses' => Expense::where('status', 'approved')->sum('amount'),
            'overdue_invoices' => MerchantInvoice::where('due_date', '<', now())->where('status', '!=', 'paid')->count()
        ];

        $recentInvoices = MerchantInvoice::with('merchant')->latest()->take(5)->get();
        $recentPayments = MerchantPayment::with('merchant')->latest()->take(5)->get();

        return view('admin.finance.dashboard', compact('stats', 'recentInvoices', 'recentPayments'));
    }

    public function cod()
    {
        $codStats = [
            'total_cod' => \App\Models\Parcel::whereNotNull('cod_amount')->sum('cod_amount'),
            'pending_cod' => \App\Models\Parcel::where('status', 'Delivered')->whereNotNull('cod_amount')->sum('cod_amount'),
            'collected_cod' => \App\Models\Parcel::where('status', 'Delivered')->where('is_scanned', true)->whereNotNull('cod_amount')->sum('cod_amount')
        ];

        return view('admin.finance.cod', compact('codStats'));
    }

    public function settlements()
    {
        $settlements = MerchantPayment::with('merchant')
            ->latest()
            ->paginate(20);

        return view('admin.finance.settlements', compact('settlements'));
    }

    public function accounting()
    {
        $accounts = CompanyAccount::where('is_active', true)->get();
        $journalEntries = JournalEntry::with('lines.account')->latest()->take(10)->get();

        return view('admin.finance.accounting', compact('accounts', 'journalEntries'));
    }

    public function expenses()
    {
        $expenses = Expense::with('approver')
            ->latest()
            ->paginate(20);

        $expenseStats = [
            'total' => Expense::where('status', 'approved')->sum('amount'),
            'pending' => Expense::where('status', 'pending')->count(),
            'approved' => Expense::where('status', 'approved')->count()
        ];

        return view('admin.finance.expenses', compact('expenses', 'expenseStats'));
    }

    public function payroll()
    {
        return view('admin.finance.payroll');
    }

    public function taxation()
    {
        return view('admin.finance.taxation');
    }

    public function banking()
    {
        return view('admin.finance.banking');
    }

    public function vouchers()
    {
        return view('admin.finance.vouchers');
    }

    public function audit()
    {
        $auditStats = [
            'total_invoices' => MerchantInvoice::count(),
            'pending_audits' => MerchantInvoice::where('status', 'sent')->count(),
            'audit_trail' => JournalEntry::where('status', 'posted')->count()
        ];

        return view('admin.finance.audit', compact('auditStats'));
    }

    public function reports()
    {
        $reportStats = [
            'total_invoices' => MerchantInvoice::count(),
            'total_payments' => MerchantPayment::count(),
            'total_expenses' => Expense::where('status', 'approved')->sum('amount'),
            'outstanding_amount' => MerchantInvoice::where('status', '!=', 'paid')->sum('total_amount')
        ];

        return view('admin.finance.reports', compact('reportStats'));
    }

    // Audit methods
    public function internalAudit()
    {
        $internalAudits = JournalEntry::where('status', 'posted')->latest()->paginate(20);
        return view('admin.finance.audit.internal-audit', compact('internalAudits'));
    }

    public function externalAudit()
    {
        return view('admin.finance.audit.external-audit');
    }

    public function compliance()
    {
        return view('admin.finance.audit.compliance');
    }

    public function sox()
    {
        return view('admin.finance.audit.sox');
    }

    public function riskAssessment()
    {
        return view('admin.finance.audit.risk-assessment');
    }

    public function accessControls()
    {
        return view('admin.finance.audit.access-controls');
    }

    // Reports methods
    public function financial()
    {
        $financialData = [
            'total_revenue' => MerchantPayment::where('status', 'completed')->sum('amount'),
            'total_expenses' => Expense::where('status', 'approved')->sum('amount'),
            'net_profit' => MerchantPayment::where('status', 'completed')->sum('amount') - Expense::where('status', 'approved')->sum('amount'),
            'outstanding_invoices' => MerchantInvoice::where('status', '!=', 'paid')->sum('total_amount')
        ];

        return view('admin.finance.reports.financial', compact('financialData'));
    }

    public function operational()
    {
        return view('admin.finance.reports.operational');
    }

    public function custom()
    {
        return view('admin.finance.reports.custom');
    }

    public function export()
    {
        return view('admin.finance.reports.export');
    }

    public function reportsDashboard()
    {
        return view('admin.finance.reports.dashboard');
    }

    public function schedule()
    {
        return view('admin.finance.reports.schedule');
    }

    // CRUD Operations for Invoices
    public function invoices()
    {
        $invoices = MerchantInvoice::with('merchant')->latest()->paginate(20);
        return view('admin.finance.invoices.index', compact('invoices'));
    }

    public function createInvoice()
    {
        $merchants = User::where('role', 'seller')->get();
        return view('admin.finance.invoices.create', compact('merchants'));
    }

    public function storeInvoice(Request $request)
    {
        $request->validate([
            'merchant_id' => 'required|exists:users,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after:invoice_date',
            'items' => 'required|array',
            'items.*.description' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0'
        ]);

        DB::transaction(function () use ($request) {
            $subtotal = collect($request->items)->sum(function ($item) {
                return $item['quantity'] * $item['unit_price'];
            });

            $invoice = MerchantInvoice::create([
                'invoice_number' => 'INV-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT),
                'merchant_id' => $request->merchant_id,
                'invoice_date' => $request->invoice_date,
                'due_date' => $request->due_date,
                'subtotal' => $subtotal,
                'total_amount' => $subtotal,
                'status' => 'draft'
            ]);

            foreach ($request->items as $item) {
                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'total_price' => $item['quantity'] * $item['unit_price']
                ]);
            }
        });

        return redirect()->route('admin.finance.invoices.index')->with('success', 'Invoice created successfully.');
    }

    public function showInvoice($id)
    {
        $invoice = MerchantInvoice::with('merchant', 'items')->findOrFail($id);
        return view('admin.finance.invoices.show', compact('invoice'));
    }

    public function editInvoice($id)
    {
        $invoice = MerchantInvoice::with('items')->findOrFail($id);
        $merchants = User::where('role', 'seller')->get();
        return view('admin.finance.invoices.edit', compact('invoice', 'merchants'));
    }

    public function updateInvoice(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:draft,sent,paid,overdue,cancelled'
        ]);

        $invoice = MerchantInvoice::findOrFail($id);
        $invoice->update(['status' => $request->status]);

        return redirect()->route('admin.finance.invoices.show', $invoice)->with('success', 'Invoice updated successfully.');
    }

    public function destroyInvoice($id)
    {
        $invoice = MerchantInvoice::findOrFail($id);
        $invoice->delete();

        return redirect()->route('admin.finance.invoices.index')->with('success', 'Invoice deleted successfully.');
    }
}
