<?php

namespace App\Http\Controllers\Admin\Marketing;

use App\Http\Controllers\Controller;
use App\Models\ComplianceDocument;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ComplianceDocumentController extends Controller
{
    public function __construct()
    {
        // Align with Spatie role names (case-sensitive)
        $this->middleware(['auth', 'role:Admin|Marketing Manager|Super Admin']);
    }

    public function index()
    {
        $documents = ComplianceDocument::latest()->paginate(10);
        return view('admin.marketing.documents.compliance.index', compact('documents'));
    }

    public function create()
    {
        return view('admin.marketing.documents.compliance.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'code' => 'required|string|max:100|unique:compliance_documents,code',
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'status' => 'required|in:draft,active,archived',
            'effective_date' => 'nullable|date',
            'expires_at' => 'nullable|date|after_or_equal:effective_date',
            'is_required' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        $data['created_by'] = Auth::id();
        $data['updated_by'] = Auth::id();

        $document = ComplianceDocument::create($data);
        $this->logAction('created', $document, $data, $request);

        return redirect()->route('admin.marketing.documents.compliance.index')
            ->with('success', 'Compliance document created successfully.');
    }

    public function show(ComplianceDocument $document)
    {
        return view('admin.marketing.documents.compliance.show', compact('document'));
    }

    public function edit(ComplianceDocument $document)
    {
        return view('admin.marketing.documents.compliance.edit', compact('document'));
    }

    public function update(Request $request, ComplianceDocument $document)
    {
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'code' => 'required|string|max:100|unique:compliance_documents,code,' . $document->id,
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'status' => 'required|in:draft,active,archived',
            'effective_date' => 'nullable|date',
            'expires_at' => 'nullable|date|after_or_equal:effective_date',
            'is_required' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        $data['updated_by'] = Auth::id();
        $document->update($data);
        $this->logAction('updated', $document, $data, $request);

        return redirect()->route('admin.marketing.documents.compliance.index')
            ->with('success', 'Compliance document updated successfully.');
    }

    public function destroy(Request $request, ComplianceDocument $document)
    {
        $document->delete();
        $this->logAction('deleted', $document, [], $request);
        return redirect()->route('admin.marketing.documents.compliance.index')
            ->with('success', 'Compliance document deleted successfully.');
    }

    protected function logAction(string $action, $model, array $changes, Request $request): void
    {
        AuditLog::create([
            'user_id' => Auth::id(),
            'action' => $action,
            'auditable_type' => get_class($model),
            'auditable_id' => $model->id,
            'changes' => $changes,
            'ip_address' => $request->ip(),
        ]);
    }
}