<?php

namespace App\Http\Controllers\Admin\Marketing;

use App\Http\Controllers\Controller;
use App\Models\DocumentTemplate;
use Illuminate\Http\Request;

class DocumentTemplateController extends Controller
{
    public function __construct()
    {
        // Use Spatie canonical role names (case-sensitive)
        $this->middleware(['auth', 'role:Admin|Marketing Manager|Super Admin']);
    }

    public function index()
    {
        $templates = DocumentTemplate::latest()->paginate(10);
        return view('admin.marketing.documents.templates.index', compact('templates'));
    }

    public function create()
    {
        return view('admin.marketing.documents.templates.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'code' => 'required|string|max:100|unique:document_templates,code',
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'content' => 'required|string',
            'status' => 'required|in:draft,active,archived',
            'version' => 'nullable|string|max:20',
            'is_global' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        $validated['created_by'] = auth()->id();

        DocumentTemplate::create($validated);

        return redirect()->route('admin.marketing.documents.templates.index')
            ->with('success', 'Template created successfully.');
    }

    public function show(DocumentTemplate $template)
    {
        return view('admin.marketing.documents.templates.show', compact('template'));
    }

    public function edit(DocumentTemplate $template)
    {
        return view('admin.marketing.documents.templates.edit', compact('template'));
    }

    public function update(Request $request, DocumentTemplate $template)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'code' => 'required|string|max:100|unique:document_templates,code,' . $template->id,
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'content' => 'required|string',
            'status' => 'required|in:draft,active,archived',
            'version' => 'nullable|string|max:20',
            'is_global' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        $validated['updated_by'] = auth()->id();

        $template->update($validated);

        return redirect()->route('admin.marketing.documents.templates.index')
            ->with('success', 'Template updated successfully.');
    }

    public function destroy(DocumentTemplate $template)
    {
        $template->delete();
        return redirect()->route('admin.marketing.documents.templates.index')
            ->with('success', 'Template deleted successfully.');
    }
}