<?php

namespace App\Http\Controllers\Admin\Marketing;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\MarketingAlert;
use Illuminate\Support\Carbon;

class MarketingController extends Controller
{
    public function dashboard()
    {
        // Basic marketing dashboard stats
        $alerts = MarketingAlert::count();
        // Placeholders for future enhancements
        $campaigns = 0;
        $zones = 0;
        $documents = 0;

        return view('admin.marketing.dashboard', compact(
            'alerts',
            'campaigns',
            'zones',
            'documents'
        ));
    }

    public function alerts()
    {
        return view('admin.marketing.alerts');
    }

    // Alerts Data Endpoint
    public function alertsData(Request $request)
    {
        $query = MarketingAlert::query();

        if ($status = $request->get('status')) {
            $query->where('status', $status);
        }
        if ($severity = $request->get('severity')) {
            $query->where('severity', $severity);
        }
        if ($category = $request->get('category')) {
            $query->where('category', $category);
        }
        if ($start = $request->get('start')) {
            try {
                $startDate = Carbon::parse($start)->startOfDay();
                $query->where('triggered_at', '>=', $startDate);
            } catch (\Exception $e) {}
        }
        if ($end = $request->get('end')) {
            try {
                $endDate = Carbon::parse($end)->endOfDay();
                $query->where('triggered_at', '<=', $endDate);
            } catch (\Exception $e) {}
        }

        $alerts = $query->orderByDesc('triggered_at')->orderByDesc('id')->get();

        return response()->json([
            'alerts' => $alerts,
            'filters' => [
                'status' => $status ?? null,
                'severity' => $severity ?? null,
                'category' => $category ?? null,
                'start' => $start ?? null,
                'end' => $end ?? null,
            ],
        ]);
    }

    // Acknowledge alert
    public function acknowledgeAlert(MarketingAlert $alert)
    {
        $alert->update(['status' => 'acknowledged', 'is_read' => true]);
        return response()->json(['success' => true]);
    }

    // Snooze alert
    public function snoozeAlert(Request $request, MarketingAlert $alert)
    {
        $minutes = (int)($request->input('minutes', 60));
        $until = Carbon::now()->addMinutes(max($minutes, 1));
        $alert->update(['status' => 'snoozed', 'snooze_until' => $until]);
        return response()->json(['success' => true, 'snooze_until' => $until->toDateTimeString()]);
    }

    public function management()
    {
        // Aggregate key marketing management metrics and recent items
        $totalAlerts = \App\Models\MarketingAlert::count();
        $unreadAlerts = \App\Models\MarketingAlert::where('is_read', false)->count();
        $recentAlerts = \App\Models\MarketingAlert::orderByDesc('triggered_at')
            ->orderByDesc('id')
            ->limit(8)
            ->get();

        // Upcoming contracts (expiring in next 60 days)
        $now = \Illuminate\Support\Carbon::now();
        $soon = $now->copy()->addDays(60);
        $expiringContracts = \App\Models\Contract::whereNotNull('end_date')
            ->whereBetween('end_date', [$now->toDateString(), $soon->toDateString()])
            ->orderBy('end_date')
            ->limit(8)
            ->get();

        // Document stats
        $templatesCount = \App\Models\DocumentTemplate::count();
        $recentTemplates = \App\Models\DocumentTemplate::orderByDesc('updated_at')
            ->limit(5)
            ->get();
        $complianceDocumentsCount = \App\Models\ComplianceDocument::count();

        // Rates stats
        $pricingCount = \App\Models\Pricing::count();
        $discountCount = \App\Models\Discount::count();
        $zoneRateCount = \App\Models\ZoneRate::count();

        return view('admin.marketing.management', compact(
            'totalAlerts',
            'unreadAlerts',
            'recentAlerts',
            'expiringContracts',
            'templatesCount',
            'recentTemplates',
            'complianceDocumentsCount',
            'pricingCount',
            'discountCount',
            'zoneRateCount'
        ));
    }

    public function pricing()
    {
        return view('admin.marketing.rates.pricing');
    }

    public function discounts()
    {
        return view('admin.marketing.rates.discounts');
    }

    public function zones()
    {
        return view('admin.marketing.rates.zones');
    }

    public function templates()
    {
        return view('admin.marketing.documents.templates');
    }

    public function contracts()
    {
        return view('admin.marketing.documents.contracts');
    }

    public function compliance()
    {
        return view('admin.marketing.documents.compliance');
    }
}
