<?php

namespace App\Http\Controllers\Admin\Marketing\Rate;

use App\Http\Controllers\Controller;
use App\Models\Discount;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class DiscountController extends Controller
{
    public function index(Request $request)
    {
        $query = Discount::query();

        if ($search = $request->input('q')) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%")
                  ->orWhere('target_code', 'like', "%{$search}%");
            });
        }

        if ($status = $request->input('status')) {
            $query->where('status', $status);
        }

        if ($type = $request->input('type')) {
            $query->where('type', $type);
        }

        if ($appliesTo = $request->input('applies_to')) {
            $query->where('applies_to', $appliesTo);
        }

        $discounts = $query->orderByDesc('id')->paginate(15)->withQueryString();

        return view('admin.marketing.rates.discounts.index', compact('discounts'));
    }

    public function create()
    {
        return view('admin.marketing.rates.discounts.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:64', 'unique:discount_rates,code'],
            'type' => ['required', Rule::in(['percentage', 'fixed'])],
            'amount' => ['required', 'numeric', 'min:0'],
            'currency' => ['nullable', 'string', 'size:3'],
            'applies_to' => ['required', Rule::in(['global', 'zone', 'merchant', 'service'])],
            'target_code' => ['nullable', 'string', 'max:64'],
            'min_order_value' => ['nullable', 'numeric', 'min:0'],
            'max_discount_value' => ['nullable', 'numeric', 'min:0'],
            'effective_from' => ['nullable', 'date'],
            'effective_to' => ['nullable', 'date', 'after_or_equal:effective_from'],
            'status' => ['required', Rule::in(['active', 'inactive', 'archived'])],
            'notes' => ['nullable', 'string'],
        ]);

        Discount::create($validated);

        return redirect()->route('admin.marketing.rates.discounts.index')
            ->with('status', 'Discount created successfully');
    }

    public function show(Discount $discount)
    {
        return view('admin.marketing.rates.discounts.show', compact('discount'));
    }

    public function edit(Discount $discount)
    {
        return view('admin.marketing.rates.discounts.edit', compact('discount'));
    }

    public function update(Request $request, Discount $discount)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:64', Rule::unique('discount_rates', 'code')->ignore($discount->id)],
            'type' => ['required', Rule::in(['percentage', 'fixed'])],
            'amount' => ['required', 'numeric', 'min:0'],
            'currency' => ['nullable', 'string', 'size:3'],
            'applies_to' => ['required', Rule::in(['global', 'zone', 'merchant', 'service'])],
            'target_code' => ['nullable', 'string', 'max:64'],
            'min_order_value' => ['nullable', 'numeric', 'min:0'],
            'max_discount_value' => ['nullable', 'numeric', 'min:0'],
            'effective_from' => ['nullable', 'date'],
            'effective_to' => ['nullable', 'date', 'after_or_equal:effective_from'],
            'status' => ['required', Rule::in(['active', 'inactive', 'archived'])],
            'notes' => ['nullable', 'string'],
        ]);

        $discount->update($validated);

        return redirect()->route('admin.marketing.rates.discounts.index')
            ->with('status', 'Discount updated successfully');
    }

    public function destroy(Discount $discount)
    {
        $discount->delete();
        return redirect()->route('admin.marketing.rates.discounts.index')
            ->with('status', 'Discount deleted successfully');
    }
}