<?php

namespace App\Http\Controllers\Admin\Marketing\Rate;

use App\Http\Controllers\Controller;
use App\Models\Pricing;
use Illuminate\Http\Request;

class PricingController extends Controller
{
    public function index(Request $request)
    {
        $query = Pricing::query();

        if ($search = $request->input('q')) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%$search%")
                  ->orWhere('code', 'like', "%$search%")
                  ->orWhere('zone', 'like', "%$search%")
                  ->orWhere('currency', 'like', "%$search%");
            });
        }

        if ($status = $request->input('status')) {
            $query->where('status', $status);
        }

        if ($zone = $request->input('zone')) {
            $query->where('zone', $zone);
        }

        $pricings = $query->orderBy('effective_from', 'desc')->paginate(15)->withQueryString();
        return view('admin.marketing.rates.pricing.index', compact('pricings'));
    }

    public function create()
    {
        return view('admin.marketing.rates.pricing.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:pricing_rates,code',
            'currency' => 'required|string|size:3',
            'base_rate' => 'required|numeric|min:0',
            'per_kg_rate' => 'nullable|numeric|min:0',
            'per_km_rate' => 'nullable|numeric|min:0',
            'min_charge' => 'nullable|numeric|min:0',
            'zone' => 'nullable|string|max:100',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after_or_equal:effective_from',
            'status' => 'required|string|in:draft,active,expired,archived',
            'notes' => 'nullable|string',
        ]);

        Pricing::create($data);
        return redirect()->route('admin.marketing.rates.pricing.index')->with('success', 'Pricing rate created.');
    }

    public function show(Pricing $pricing)
    {
        return view('admin.marketing.rates.pricing.show', compact('pricing'));
    }

    public function edit(Pricing $pricing)
    {
        return view('admin.marketing.rates.pricing.edit', compact('pricing'));
    }

    public function update(Request $request, Pricing $pricing)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:pricing_rates,code,' . $pricing->id,
            'currency' => 'required|string|size:3',
            'base_rate' => 'required|numeric|min:0',
            'per_kg_rate' => 'nullable|numeric|min:0',
            'per_km_rate' => 'nullable|numeric|min:0',
            'min_charge' => 'nullable|numeric|min:0',
            'zone' => 'nullable|string|max:100',
            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date|after_or_equal:effective_from',
            'status' => 'required|string|in:draft,active,expired,archived',
            'notes' => 'nullable|string',
        ]);

        $pricing->update($data);
        return redirect()->route('admin.marketing.rates.pricing.index')->with('success', 'Pricing rate updated.');
    }

    public function destroy(Pricing $pricing)
    {
        $pricing->delete();
        return redirect()->route('admin.marketing.rates.pricing.index')->with('success', 'Pricing rate deleted.');
    }
}