<?php

namespace App\Http\Controllers\Admin\Marketing\Rate;

use App\Http\Controllers\Controller;
use App\Models\ZoneRate;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ZoneController extends Controller
{
    public function index(Request $request)
    {
        $query = ZoneRate::query();

        if ($search = $request->input('q')) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%")
                  ->orWhere('zone', 'like', "%{$search}%");
            });
        }

        if ($status = $request->input('status')) {
            $query->where('status', $status);
        }

        if ($zone = $request->input('zone')) {
            $query->where('zone', 'like', "%{$zone}%");
        }

        if ($currency = $request->input('currency')) {
            $query->where('currency', $currency);
        }

        $zones = $query->orderByDesc('id')->paginate(15)->withQueryString();

        return view('admin.marketing.rates.zones.index', compact('zones'));
    }

    public function create()
    {
        return view('admin.marketing.rates.zones.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:64', 'unique:zone_rates,code'],
            'zone' => ['required', 'string', 'max:64'],
            'currency' => ['required', 'string', 'size:3'],
            'base_rate' => ['required', 'numeric', 'min:0'],
            'additional_per_kg' => ['nullable', 'numeric', 'min:0'],
            'remote_area_fee' => ['nullable', 'numeric', 'min:0'],
            'fuel_surcharge_percent' => ['nullable', 'numeric', 'min:0'],
            'effective_from' => ['nullable', 'date'],
            'effective_to' => ['nullable', 'date', 'after_or_equal:effective_from'],
            'status' => ['required', Rule::in(['active', 'inactive', 'archived'])],
            'notes' => ['nullable', 'string'],
        ]);

        ZoneRate::create($validated);

        return redirect()->route('admin.marketing.rates.zones.index')
            ->with('status', 'Zone rate created successfully');
    }

    public function show(ZoneRate $zoneRate)
    {
        return view('admin.marketing.rates.zones.show', ['zone' => $zoneRate]);
    }

    public function edit(ZoneRate $zoneRate)
    {
        return view('admin.marketing.rates.zones.edit', ['zone' => $zoneRate]);
    }

    public function update(Request $request, ZoneRate $zoneRate)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:64', Rule::unique('zone_rates', 'code')->ignore($zoneRate->id)],
            'zone' => ['required', 'string', 'max:64'],
            'currency' => ['required', 'string', 'size:3'],
            'base_rate' => ['required', 'numeric', 'min:0'],
            'additional_per_kg' => ['nullable', 'numeric', 'min:0'],
            'remote_area_fee' => ['nullable', 'numeric', 'min:0'],
            'fuel_surcharge_percent' => ['nullable', 'numeric', 'min:0'],
            'effective_from' => ['nullable', 'date'],
            'effective_to' => ['nullable', 'date', 'after_or_equal:effective_from'],
            'status' => ['required', Rule::in(['active', 'inactive', 'archived'])],
            'notes' => ['nullable', 'string'],
        ]);

        $zoneRate->update($validated);

        return redirect()->route('admin.marketing.rates.zones.index')
            ->with('status', 'Zone rate updated successfully');
    }

    public function destroy(ZoneRate $zoneRate)
    {
        $zoneRate->delete();
        return redirect()->route('admin.marketing.rates.zones.index')
            ->with('status', 'Zone rate deleted successfully');
    }
}