<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Parcel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class MerchantController extends Controller
{
    public function index(Request $request)
    {
        $query = User::where('role', 'seller')
            ->withCount('parcels');

        // Apply search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%");
            });
        }

        // Apply status filter
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->whereNotNull('email_verified_at');
            } elseif ($request->status === 'inactive') {
                $query->whereNull('email_verified_at');
            }
        }

        $merchants = $query->latest()->paginate(20);
        
        // Get statistics
        $stats = [
            'total' => User::where('role', 'seller')->count(),
            'active' => User::where('role', 'seller')->whereNotNull('email_verified_at')->count(),
            'total_parcels' => Parcel::whereIn('seller_id', User::where('role', 'seller')->pluck('id'))->count(),
            'total_revenue' => Parcel::whereIn('seller_id', User::where('role', 'seller')->pluck('id'))
                ->where('status', 'Delivered')
                ->sum('cod_amount'),
        ];
            
        return view('admin.merchants.index', compact('merchants', 'stats'));
    }

    public function create()
    {
        return view('admin.merchants.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'company_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'phone' => 'nullable|string|max:20',
        ]);

        $validated['password'] = bcrypt($validated['password']);
        $validated['role'] = 'seller';

        User::create($validated);

        return redirect()->route('admin.merchants.index')
            ->with('success', 'Merchant added successfully.');
    }

    public function show(User $merchant)
    {
        $merchant->load(['parcels' => function($query) {
            $query->latest()->limit(10);
        }]);
        
        $parcelStats = [
            'total' => $merchant->parcels()->count(),
            'pending' => $merchant->parcels()->where('status', 'Pending')->count(),
            'delivered' => $merchant->parcels()->where('status', 'Delivered')->count(),
            'returned' => $merchant->parcels()->where('status', 'Returned')->count(),
        ];
        
        return view('admin.merchants.show', compact('merchant', 'parcelStats'));
    }

    public function edit(User $merchant)
    {
        return view('admin.merchants.edit', compact('merchant'));
    }

    public function update(Request $request, User $merchant)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'company_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $merchant->id,
            'password' => 'nullable|string|min:8',
            'phone' => 'nullable|string|max:20',
        ]);

        if (!empty($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        } else {
            unset($validated['password']);
        }

        $merchant->update($validated);

        return redirect()->route('admin.merchants.index')
            ->with('success', 'Merchant updated successfully.');
    }

    public function destroy(User $merchant)
    {
        // Check if merchant has parcels
        if ($merchant->parcels()->count() > 0) {
            return back()->with('error', 'Cannot delete merchant with existing parcels.');
        }

        $merchant->delete();

        return redirect()->route('admin.merchants.index')
            ->with('success', 'Merchant deleted successfully.');
    }

    /**
     * Approve a merchant account.
     */
    public function approve(User $merchant)
    {
        // Only seller accounts are approvable
        if ($merchant->role !== 'seller') {
            return back()->with('error', 'Only seller accounts can be approved.');
        }

        if ($merchant->is_approved) {
            return back()->with('success', 'Merchant already approved.');
        }

        $merchant->is_approved = true;
        $merchant->approved_at = now();
        $merchant->save();

        return back()->with('success', 'Merchant approved successfully.');
    }
}
