<?php

namespace App\Http\Controllers\Admin\Rider;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Rider;
use App\Models\User;

class RiderController extends Controller
{
    public function dashboard()
    {
        $totalRiders = Rider::count();
        $activeRiders = Rider::where('status', 'active')->count();
        // Riders on duty determined by having active parcels in progress
        $ridersOnDuty = Rider::whereHas('parcels', function($query) {
            $query->whereIn('status', ['Picked Up', 'Out for Delivery']);
        })->count();
        $deliveriesToday = Rider::whereHas('parcels', function($query) {
            $query->where('status', 'Delivered')->whereDate('updated_at', today());
        })->count();

        $recentDeliveries = Rider::with(['parcels' => function($query) {
            $query->where('status', 'Delivered')->latest()->take(5);
        }])->take(10)->get();

        return view('admin.riders.dashboard', compact(
            'totalRiders',
            'activeRiders',
            'ridersOnDuty',
            'deliveriesToday',
            'recentDeliveries'
        ));
    }

    public function operations()
    {
        $riders = Rider::with(['parcels' => function($query) {
            $query->whereIn('status', ['Picked Up', 'Out for Delivery', 'Delivered']);
        }])->latest()->paginate(15);

        // Compute stats without relying on nonexistent current_status column
        $activeDeliveries = Rider::whereHas('parcels', function($query) {
            $query->whereIn('status', ['Picked Up', 'Out for Delivery']);
        })->count();

        $availableRiders = Rider::whereDoesntHave('parcels', function($query) {
            $query->whereIn('status', ['Picked Up', 'Out for Delivery']);
        })->count();

        $inTransitCount = \App\Models\Parcel::whereIn('status', ['Picked Up', 'Out for Delivery'])->count();

        $completedToday = \App\Models\Parcel::where('status', 'Delivered')
            ->whereDate('updated_at', today())
            ->count();

        return view('admin.riders.operations', compact(
            'riders',
            'activeDeliveries',
            'availableRiders',
            'inTransitCount',
            'completedToday'
        ));
    }

    public function index()
    {
        $riders = Rider::latest()->paginate(15);
        return view('admin.riders.index', compact('riders'));
    }

    public function create()
    {
        return view('admin.riders.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:riders,email',
            'phone' => 'required|string|max:20',
            'license_number' => 'required|string|unique:riders,license_number',
            'vehicle_type' => 'required|string|max:255',
            'status' => 'required|in:active,inactive',
        ]);

        Rider::create($request->all());

        return redirect()->route('admin.riders.index')->with('success', 'Rider created successfully.');
    }

    public function show(Rider $rider)
    {
        $rider->load(['parcels' => function($query) {
            $query->latest()->take(10);
        }]);

        return view('admin.riders.show', compact('rider'));
    }

    public function edit(Rider $rider)
    {
        return view('admin.riders.edit', compact('rider'));
    }

    public function update(Request $request, Rider $rider)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:riders,email,' . $rider->id,
            'phone' => 'required|string|max:20',
            'license_number' => 'required|string|unique:riders,license_number,' . $rider->id,
            'vehicle_type' => 'required|string|max:255',
            'status' => 'required|in:active,inactive',
        ]);

        $rider->update($request->all());

        return redirect()->route('admin.riders.index')->with('success', 'Rider updated successfully.');
    }

    public function destroy(Rider $rider)
    {
        $rider->delete();

        return redirect()->route('admin.riders.index')->with('success', 'Rider deleted successfully.');
    }
}
