<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SupportController extends Controller
{
    public function tickets()
    {
        $tickets = SupportTicket::with(['user', 'assignedUser'])
            ->latest()
            ->paginate(20);

        $stats = [
            'total' => SupportTicket::count(),
            'open' => SupportTicket::where('status', 'open')->count(),
            'pending' => SupportTicket::where('status', 'pending')->count(),
            'resolved' => SupportTicket::where('status', 'resolved')->count(),
            'overdue' => SupportTicket::where('sla_deadline', '<', now())->count()
        ];

        return view('admin.support.tickets', compact('tickets', 'stats'));
    }

    public function open()
    {
        $tickets = SupportTicket::with(['user', 'assignedUser'])
            ->where('status', 'open')
            ->latest()
            ->paginate(20);

        return view('admin.support.tickets.open', compact('tickets'));
    }

    public function pending()
    {
        $tickets = SupportTicket::with(['user', 'assignedUser'])
            ->where('status', 'pending')
            ->latest()
            ->paginate(20);

        return view('admin.support.tickets.pending', compact('tickets'));
    }

    public function resolved()
    {
        $tickets = SupportTicket::with(['user', 'assignedUser'])
            ->where('status', 'resolved')
            ->latest()
            ->paginate(20);

        return view('admin.support.tickets.resolved', compact('tickets'));
    }

    public function escalated()
    {
        $tickets = SupportTicket::with(['user', 'assignedUser'])
            ->where('status', 'escalated')
            ->latest()
            ->paginate(20);

        return view('admin.support.tickets.escalated', compact('tickets'));
    }

    public function show($id)
    {
        $ticket = SupportTicket::with(['user', 'assignedUser', 'replies.user'])->findOrFail($id);

        return view('admin.support.tickets.show', compact('ticket'));
    }

    public function create()
    {
        $users = User::whereIn('role', ['seller', 'admin'])->get();

        return view('admin.support.tickets.create', compact('users'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'category' => 'required|in:technical,billing,shipping,complaint,feature_request,other',
            'user_id' => 'required|exists:users,id',
            'assigned_to' => 'nullable|exists:users,id'
        ]);

        $ticketNumber = 'TKT-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);

        $slaDeadline = now()->addMinutes($request->priority === 'urgent' ? 240 : ($request->priority === 'high' ? 480 : 1440));

        SupportTicket::create([
            'ticket_number' => $ticketNumber,
            'title' => $request->title,
            'description' => $request->description,
            'priority' => $request->priority,
            'category' => $request->category,
            'user_id' => $request->user_id,
            'assigned_to' => $request->assigned_to,
            'sla_deadline' => $slaDeadline,
            'status' => 'open'
        ]);

        return redirect()->route('admin.support.tickets')->with('success', 'Support ticket created successfully.');
    }

    public function edit($id)
    {
        $ticket = SupportTicket::findOrFail($id);
        $users = User::whereIn('role', ['seller', 'admin'])->get();

        return view('admin.support.tickets.edit', compact('ticket', 'users'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'category' => 'required|in:technical,billing,shipping,complaint,feature_request,other',
            'status' => 'required|in:open,pending,in_progress,resolved,closed,escalated',
            'assigned_to' => 'nullable|exists:users,id'
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $ticket->update($request->except('user_id'));

        if ($request->status === 'resolved' && !$ticket->resolved_at) {
            $ticket->update(['resolved_at' => now()]);
        }

        return redirect()->route('admin.support.tickets.show', $ticket)->with('success', 'Ticket updated successfully.');
    }

    public function assign(Request $request, $id)
    {
        $request->validate([
            'assigned_to' => 'required|exists:users,id'
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $ticket->update([
            'assigned_to' => $request->assigned_to,
            'status' => 'in_progress'
        ]);

        return redirect()->back()->with('success', 'Ticket assigned successfully.');
    }

    public function addReply(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string',
            'is_internal' => 'boolean'
        ]);

        $ticket = SupportTicket::findOrFail($id);

        SupportTicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'is_internal' => $request->is_internal ?? false
        ]);

        return redirect()->back()->with('success', 'Reply added successfully.');
    }

    public function categories()
    {
        $categories = [
            'technical' => SupportTicket::where('category', 'technical')->count(),
            'billing' => SupportTicket::where('category', 'billing')->count(),
            'shipping' => SupportTicket::where('category', 'shipping')->count(),
            'complaint' => SupportTicket::where('category', 'complaint')->count(),
            'feature_request' => SupportTicket::where('category', 'feature_request')->count(),
            'other' => SupportTicket::where('category', 'other')->count()
        ];

        return view('admin.support.tickets.categories', compact('categories'));
    }

    public function sla()
    {
        $slaStats = [
            'within_sla' => SupportTicket::whereColumn('resolved_at', '<=', 'sla_deadline')->whereNotNull('resolved_at')->count(),
            'breached_sla' => SupportTicket::whereColumn('resolved_at', '>', 'sla_deadline')->whereNotNull('resolved_at')->count(),
            'overdue' => SupportTicket::where('sla_deadline', '<', now())->where('status', '!=', 'resolved')->count(),
            'total' => SupportTicket::count()
        ];

        return view('admin.support.tickets.sla', compact('slaStats'));
    }

    public function destroy($id)
    {
        $ticket = SupportTicket::findOrFail($id);
        $ticket->delete();

        return redirect()->route('admin.support.tickets')->with('success', 'Ticket deleted successfully.');
    }
}
