<?php

namespace App\Http\Controllers;

use App\Models\Parcel;
use App\Models\Complaint;
use App\Models\Payout;
use App\Models\News;
use App\Models\User;
use App\Models\Rider;
use App\Models\Branch;
use App\Models\CodCollection;
use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Show the seller dashboard.
     */
    public function sellerDashboard()
    {
        $user = Auth::user();
        
        // Get statistics
        $totalParcels = Parcel::where('seller_id', $user->id)->count();
        $deliveredParcels = Parcel::where('seller_id', $user->id)
            ->where('status', 'Delivered')
            ->count();
        $pendingParcels = Parcel::where('seller_id', $user->id)
            ->where('status', 'Pending')
            ->count();
        $inTransitParcels = Parcel::where('seller_id', $user->id)
            ->whereIn('status', ['Picked Up', 'In Transit', 'Out for Delivery'])
            ->count();
        
        // Calculate wallet balance (delivered COD - paid payouts)
        $totalCOD = Parcel::where('seller_id', $user->id)
            ->where('status', 'Delivered')
            ->sum('cod_amount');
        $totalPayouts = Payout::where('seller_id', $user->id)
            ->where('status', 'Paid')
            ->sum('amount');
        $walletBalance = $totalCOD - $totalPayouts;
        
        // Recent parcels
        $recentParcels = Parcel::where('seller_id', $user->id)
            ->with('rider')
            ->latest()
            ->take(10)
            ->get();
        
        // Recent complaints
        $recentComplaints = Complaint::where('seller_id', $user->id)
            ->with('parcel')
            ->latest()
            ->take(5)
            ->get();
        
        // News
        $news = News::latest()->take(3)->get();
        
        return view('seller.dashboard', compact(
            'totalParcels',
            'deliveredParcels',
            'pendingParcels',
            'inTransitParcels',
            'walletBalance',
            'recentParcels',
            'recentComplaints',
            'news'
        ));
    }

    /**
     * Show the admin dashboard.
     */
    public function adminDashboard(Request $request)
    {
        $period = $request->get('period');
        $q = trim((string) $request->get('q'));
        $startDate = null;
        if (in_array($period, ['7', '30', '90'])) {
            $startDate = now()->subDays((int) $period)->startOfDay();
        }

        $parcelsQuery = Parcel::when($startDate, function ($query) use ($startDate) {
            $query->where('created_at', '>=', $startDate);
        });

        $totalParcels = $parcelsQuery->count();
        $deliveredParcels = (clone $parcelsQuery)->where('status', 'Delivered')->count();
        $pendingParcels = (clone $parcelsQuery)->where('status', 'Pending')->count();
        $inTransitParcels = (clone $parcelsQuery)->whereIn('status', ['Picked Up', 'In Transit', 'Out for Delivery'])->count();

        $totalSellers = User::where('role', 'seller')->count();
        $totalRiders = Rider::count();
        $totalBranches = Branch::count();

        $codQuery = Parcel::when($startDate, function ($query) use ($startDate) {
            $query->where('updated_at', '>=', $startDate);
        });

        $totalCODCollected = (clone $codQuery)->where('status', 'Delivered')->sum('cod_amount');
        $pendingCOD = CodCollection::sum('pending_amount'); // This might not be affected by date filter
        $reconciledCOD = CodCollection::whereIn('status', ['completed', 'verified'])->sum('collected_amount'); // This might not be affected by date filter

        $totalExpenses = Expense::where('status', 'approved')
            ->when($startDate, function ($query) use ($startDate) {
                $query->where('expense_date', '>=', $startDate);
            })
            ->sum('amount');

        $recentParcelsQuery = Parcel::with(['seller', 'rider'])
            ->when($startDate, function ($query) use ($startDate) {
                $query->where('created_at', '>=', $startDate);
            });

        if ($q !== '') {
            $recentParcelsQuery->where(function ($sub) use ($q) {
                $sub->where('tracking_id', 'like', "%$q%")
                    ->orWhere('recipient_name', 'like', "%$q%")
                    ->orWhereHas('seller', function ($sellerQuery) use ($q) {
                        $sellerQuery->where('company_name', 'like', "%$q%");
                    });
            });
        }
        $recentParcels = $recentParcelsQuery->latest()->take(10)->get();

        $activeComplaints = Complaint::whereIn('status', ['Open', 'In Progress'])
            ->when($startDate, function ($query) use ($startDate) {
                $query->where('created_at', '>=', $startDate);
            })
            ->with(['parcel', 'seller'])
            ->latest()
            ->take(5)
            ->get();

        return view('admin.dashboard', compact(
            'totalParcels',
            'deliveredParcels',
            'totalSellers',
            'totalRiders',
            'totalBranches',
            'pendingParcels',
            'inTransitParcels',
            'totalCODCollected',
            'pendingCOD',
            'reconciledCOD',
            'totalExpenses',
            'recentParcels',
            'activeComplaints',
            'period',
            'q'
        ));
    }

    public function branchDashboard()
    {
        $totalParcels = Parcel::count();
        $assignedShipments = Parcel::whereNotNull('assigned_rider_id')->count();
        $deliveredToday = Parcel::where('status', 'Delivered')
            ->whereDate('updated_at', today())
            ->count();
        $failedDeliveries = Parcel::where('status', 'Failed')->count();
        
        $recentParcels = Parcel::with(['seller', 'rider'])
            ->latest()
            ->take(10)
            ->get();
        
        return view('admin.branch.dashboard', compact(
            'totalParcels',
            'assignedShipments',
            'deliveredToday',
            'failedDeliveries',
            'recentParcels'
        ));
    }

    public function financeDashboard()
    {
        $totalCOD = Parcel::where('status', 'Delivered')->sum('cod_amount');
        $todayCOD = Parcel::where('status', 'Delivered')
            ->whereDate('updated_at', today())
            ->sum('cod_amount');
        $pendingPayouts = Payout::where('status', 'Pending')->sum('amount');
        $paidPayouts = Payout::where('status', 'Paid')->sum('amount');
        
        $recentPayouts = Payout::with('seller')
            ->latest()
            ->take(10)
            ->get();
        
        $dailyCollections = [];
        
        return view('admin.finance.dashboard', compact(
            'totalCOD',
            'todayCOD',
            'pendingPayouts',
            'paidPayouts',
            'recentPayouts',
            'dailyCollections'
        ));
    }

    public function marketingDashboard()
    {
        $totalSellers = User::where('role', 'seller')->count();
        $newSellersThisMonth = User::where('role', 'seller')
            ->whereMonth('created_at', now()->month)
            ->count();
        $activeSellers = Parcel::distinct('seller_id')
            ->whereMonth('created_at', now()->month)
            ->count('seller_id');
        $totalParcels = Parcel::count();
        
        $news = News::latest()->take(5)->get();
        
        return view('admin.marketing.dashboard', compact(
            'totalSellers',
            'newSellersThisMonth',
            'activeSellers',
            'totalParcels',
            'news'
        ));
    }

    public function supportDashboard()
    {
        $openComplaints = Complaint::where('status', 'Open')->count();
        $inProgressComplaints = Complaint::where('status', 'In Progress')->count();
        $resolvedToday = Complaint::where('status', 'Resolved')
            ->whereDate('updated_at', today())
            ->count();
        $totalComplaints = Complaint::count();
        
        $recentComplaints = Complaint::with(['parcel', 'seller'])
            ->latest()
            ->take(10)
            ->get();
        
        return view('admin.support.dashboard', compact(
            'openComplaints',
            'inProgressComplaints',
            'resolvedToday',
            'totalComplaints',
            'recentComplaints'
        ));
    }

    public function riderManagerDashboard()
    {
        $totalRiders = Rider::count();
        $activeRiders = Rider::where('status', 'active')->count();
        $ridersOnDuty = Parcel::whereIn('status', ['Picked Up', 'Out for Delivery'])
            ->distinct('assigned_rider_id')
            ->count('assigned_rider_id');
        $deliveriesToday = Parcel::where('status', 'Delivered')
            ->whereDate('updated_at', today())
            ->count();
        
        $riders = Rider::withCount([
            'parcels as total_deliveries' => function($query) {
                $query->where('status', 'Delivered');
            }
        ])->latest()->take(10)->get();
        
        return view('admin.riders.dashboard', compact(
            'totalRiders',
            'activeRiders',
            'ridersOnDuty',
            'deliveriesToday',
            'riders'
        ));
    }

    public function hrDashboard()
    {
        $totalEmployees = User::whereIn('role', ['admin', 'super-admin', 'branch-operator', 'finance', 'marketing-manager', 'support', 'hr'])->count();
        $totalRiders = Rider::count();
        $newHiresThisMonth = User::whereMonth('created_at', now()->month)->count();
        $totalSellers = User::where('role', 'seller')->count();
        
        $recentUsers = User::latest()->take(10)->get();
        
        return view('admin.hr.dashboard', compact(
            'totalEmployees',
            'totalRiders',
            'newHiresThisMonth',
            'totalSellers',
            'recentUsers'
        ));
    }
}
