<?php

namespace App\Http\Controllers;

use App\Models\Parcel;
use App\Models\User;
use App\Models\Rider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ParcelController extends Controller
{
    /**
     * Display a listing of the parcels.
     */
    public function index(Request $request)
    {
        $query = Parcel::with(['seller', 'rider']);

        // Check if admin route (canonical Spatie role names)
        $isAdmin = Auth::user()->hasAnyRole(['Admin', 'Super Admin', 'Branch Operator']);
        
        // Filter by seller if not admin
        if (!$isAdmin) {
            $query->where('seller_id', Auth::id());
        }

        // Apply filters
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('tracking_id', 'like', "%{$search}%")
                  ->orWhere('recipient_name', 'like', "%{$search}%")
                  ->orWhere('contact', 'like', "%{$search}%");
            });
        }

        // Admin specific filters
        if ($isAdmin) {
            if ($request->filled('city')) {
                $query->where('city', 'like', "%{$request->city}%");
            }

            if ($request->filled('zone')) {
                $query->where('zone', 'like', "%{$request->zone}%");
            }

            if ($request->filled('seller_id')) {
                $query->where('seller_id', $request->seller_id);
            }

            if ($request->filled('rider_id')) {
                $query->where('assigned_rider_id', $request->rider_id);
            }

            if ($request->filled('date_from')) {
                $query->whereDate('booking_date', '>=', $request->date_from);
            }

            if ($request->filled('date_to')) {
                $query->whereDate('booking_date', '<=', $request->date_to);
            }
        }

        $parcels = $query->latest()->paginate(20);

        // For admin view, add extra data
        if ($isAdmin && request()->is('admin/*')) {
            $sellers = User::where('role', 'seller')->select('id', 'name', 'company_name')->get();
            $riders = Rider::select('id', 'name')->get();
            
            // Get status counts
            $statusCounts = Parcel::select('status', DB::raw('count(*) as count'))
                ->groupBy('status')
                ->pluck('count', 'status')
                ->toArray();

            return view('admin.parcels.index', compact('parcels', 'sellers', 'riders', 'statusCounts'));
        }

        return view('parcels.index', compact('parcels'));
    }

    /**
     * Show the form for creating a new parcel.
     */
    public function create()
    {
        return view('parcels.create');
    }

    /**
     * Store a newly created parcel in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'order_type' => 'required|string',
            'order_reference_number' => 'required|string|min:3',
            'cod_amount' => 'required|numeric|min:0',
            'booking_date' => 'required|date',
            'recipient_name' => 'required|string|min:3',
            'contact' => 'required|string|min:10',
            'city' => 'required|string',
            'address' => 'required|string|min:10',
            'items' => 'required|integer|min:1',
            'pickup_city' => 'required|string',
            'pickup_address' => 'required|string',
            'return_city' => 'required|string',
            'return_address' => 'required|string',
            'booking_weight' => 'required|numeric|min:0.1',
            'zone' => 'required|string',
            'order_detail' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        $validated['tracking_id'] = Parcel::generateTrackingId();
        $validated['seller_id'] = Auth::id();
        $validated['status'] = 'Pending';

        $parcel = Parcel::create($validated);

        return redirect()->route('parcels.show', $parcel)
            ->with('success', 'Parcel booked successfully! Tracking ID: ' . $parcel->tracking_id);
    }

    /**
     * Display the specified parcel.
     */
    public function show(Parcel $parcel)
    {
        // Check authorization
        if (!Auth::user()->hasAnyRole(['Admin', 'Super Admin', 'Branch Operator']) 
            && $parcel->seller_id !== Auth::id()) {
            abort(403);
        }

        $parcel->load(['seller', 'rider', 'complaints']);

        return view('parcels.show', compact('parcel'));
    }

    /**
     * Track a parcel by tracking ID.
     */
    public function track(Request $request)
    {
        $request->validate([
            'tracking_id' => 'required|string',
        ]);

        $parcel = Parcel::where('tracking_id', $request->tracking_id)
            ->with(['seller', 'rider'])
            ->first();

        if (!$parcel) {
            return back()->with('error', 'Parcel not found with tracking ID: ' . $request->tracking_id);
        }

        return view('parcels.track', compact('parcel'));
    }

    /**
     * Show the form for editing the specified parcel.
     */
    public function edit(Parcel $parcel)
    {
        // Only admins can edit parcels
        if (!Auth::user()->hasAnyRole(['Admin', 'Super Admin', 'Branch Operator'])) {
            abort(403);
        }

        $sellers = User::where('role', 'seller')->select('id', 'name', 'company_name')->get();
        $riders = Rider::select('id', 'name')->get();

        return view('admin.parcels.edit', compact('parcel', 'sellers', 'riders'));
    }

    /**
     * Update the specified parcel in storage.
     */
    public function update(Request $request, Parcel $parcel)
    {
        // Only admins can update parcels
        if (!Auth::user()->hasAnyRole(['Admin', 'Super Admin', 'Branch Operator'])) {
            abort(403);
        }

        $validated = $request->validate([
            'status' => 'sometimes|string',
            'assigned_rider_id' => 'sometimes|nullable|exists:riders,id',
            'notes' => 'sometimes|nullable|string',
            'recipient_name' => 'sometimes|string|min:3',
            'contact' => 'sometimes|string|min:10',
            'city' => 'sometimes|string',
            'address' => 'sometimes|string|min:10',
            'cod_amount' => 'sometimes|numeric|min:0',
            'zone' => 'sometimes|string',
        ]);

        $parcel->update($validated);

        return back()->with('success', 'Parcel updated successfully!');
    }

    /**
     * Remove the specified parcel from storage.
     */
    public function destroy(Parcel $parcel)
    {
        // Only admins can delete parcels
        if (!Auth::user()->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403);
        }

        $trackingId = $parcel->tracking_id;
        $parcel->delete();

        return redirect()->route('admin.parcels.index')
            ->with('success', "Parcel {$trackingId} has been deleted successfully!");
    }

    /**
     * Bulk update parcel status.
     */
    public function bulkUpdate(Request $request)
    {
        // Only admins can bulk update parcels
        if (!Auth::user()->hasAnyRole(['Admin', 'Super Admin', 'Branch Operator'])) {
            abort(403);
        }

        $request->validate([
            'parcel_ids' => 'required|string',
            'bulk_status' => 'required|string',
        ]);

        $parcelIds = explode(',', $request->parcel_ids);
        
        $updated = Parcel::whereIn('id', $parcelIds)->update([
            'status' => $request->bulk_status,
        ]);

        return back()->with('success', "{$updated} parcel(s) updated to status: {$request->bulk_status}");
    }
}
