<?php

namespace App\Http\Controllers\Seller;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Complaint;

class ReportController extends Controller
{
    public function orders()
    {
        $weeklyOrders = Order::where('user_id', auth()->id())
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->get();

        $monthlyOrders = Order::where('user_id', auth()->id())
            ->selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, COUNT(*) as count')
            ->groupBy('year', 'month')
            ->get();

        return view('seller.reports.orders', compact('weeklyOrders', 'monthlyOrders'));
    }

    public function financial()
    {
        // Basic financial metrics for seller
        $totalCod = Order::where('user_id', auth()->id())
            ->whereNotNull('cod_amount')
            ->sum('cod_amount');
        $deliveredCod = Order::where('user_id', auth()->id())
            ->where('status', 'Delivered')
            ->sum('cod_amount');
        $pendingCod = $totalCod - $deliveredCod;

        return view('seller.reports.financial', compact('totalCod', 'deliveredCod', 'pendingCod'));
    }

    public function amounts()
    {
        $weeklyAmounts = Order::where('user_id', auth()->id())
            ->selectRaw('DATE(created_at) as date, SUM(cod_amount) as amount')
            ->groupBy('date')
            ->get();

        $monthlyAmounts = Order::where('user_id', auth()->id())
            ->selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, SUM(cod_amount) as amount')
            ->groupBy('year', 'month')
            ->get();

        return view('seller.reports.amounts', compact('weeklyAmounts', 'monthlyAmounts'));
    }

    public function status(Request $request)
    {
        $query = Order::where('user_id', auth()->id());

        // Status/Tab filtering
        $status = $request->get('tab');
        if ($status) {
            $query->where('status', $status);
        }

        // Search filtering
        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('tracking_id', 'like', "%$search%")
                  ->orWhere('recipient_name', 'like', "%$search%");
            });
        }

        // Date filtering
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->get('from_date'));
        }
        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->get('to_date'));
        }

        $orders = $query->latest()->paginate(10);

        // Counts for tabs
        $allCount = Order::where('user_id', auth()->id())->count();
        $bookedCount = Order::where('user_id', auth()->id())->where('status', 'Booked')->count();
        $unbookedCount = Order::where('user_id', auth()->id())->where('status', 'Pending')->count();
        $inTransitCount = Order::where('user_id', auth()->id())->where('status', 'In Transit')->count();

        return view('seller.reports.status', compact('orders', 'allCount', 'bookedCount', 'unbookedCount', 'inTransitCount'));
    }

    public function instock(Request $request)
    {
        $query = Order::where('user_id', auth()->id())->where('status', 'In Warehouse');

        // Search filtering
        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('tracking_id', 'like', "%$search%")
                  ->orWhere('recipient_name', 'like', "%$search%");
            });
        }

        // Date filtering
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->get('from_date'));
        }
        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->get('to_date'));
        }

        $orders = $query->latest()->paginate(10);

        return view('seller.reports.instock', compact('orders'));
    }

    public function pickup(Request $request)
    {
        $query = Order::where('user_id', auth()->id())->where('status', 'Picked Up');

        // Search filtering
        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('tracking_id', 'like', "%$search%")
                  ->orWhere('recipient_name', 'like', "%$search%");
            });
        }

        // Date filtering
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->get('from_date'));
        }
        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->get('to_date'));
        }

        $orders = $query->latest()->paginate(10);

        return view('seller.reports.pickup', compact('orders'));
    }

    public function destination(Request $request)
    {
        $query = Order::where('user_id', auth()->id());

        // Status/Tab filtering
        $status = $request->get('tab');
        if ($status) {
            $query->where('status', $status);
        }

        // Search filtering
        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('tracking_id', 'like', "%$search%")
                  ->orWhere('recipient_name', 'like', "%$search%");
            });
        }

        // Date filtering
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->get('from_date'));
        }
        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->get('to_date'));
        }

        // City filtering
        if ($request->filled('city')) {
            $query->where('recipient_city', $request->get('city'));
        }

        $orders = $query->latest()->paginate(10);

        // Counts for tabs
        $allCount = Order::where('user_id', auth()->id())->count();
        $bookedCount = Order::where('user_id', auth()->id())->where('status', 'Booked')->count();
        $unbookedCount = Order::where('user_id', auth()->id())->where('status', 'Pending')->count();
        $inTransitCount = Order::where('user_id', auth()->id())->where('status', 'In Transit')->count();

        return view('seller.reports.destination', compact('orders', 'allCount', 'bookedCount', 'unbookedCount', 'inTransitCount'));
    }

    public function complaints()
    {
        $complaints = Complaint::with(['parcel', 'seller'])
            ->where(function ($q) {
                $q->where('seller_id', auth()->id())
                  ->orWhere('user_id', auth()->id());
            })
            ->latest()
            ->paginate(20);

        return view('seller.reports.complaints', compact('complaints'));
    }

    public function downloadOrdersPdf(Request $request)
    {
        $orders = Order::where('user_id', auth()->id())->get();
        $pdf = Pdf::loadView('seller.reports.orders_pdf', compact('orders'));
        return $pdf->download('order-report.pdf');
    }
}
