<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class MerchantInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_number',
        'merchant_id',
        'invoice_date',
        'due_date',
        'subtotal',
        'tax_amount',
        'discount_amount',
        'total_amount',
        'status',
        'notes'
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'due_date' => 'date'
    ];

    public function merchant()
    {
        return $this->belongsTo(User::class, 'merchant_id');
    }

    public function items()
    {
        return $this->hasMany(InvoiceItem::class, 'invoice_id');
    }

    public function payments()
    {
        return $this->hasMany(MerchantPayment::class, 'merchant_id', 'merchant_id');
    }
}

class InvoiceItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_id',
        'description',
        'quantity',
        'unit_price',
        'total_price'
    ];

    public function invoice()
    {
        return $this->belongsTo(MerchantInvoice::class, 'invoice_id');
    }
}

class MerchantPayment extends Model
{
    use HasFactory;

    protected $fillable = [
        'payment_number',
        'merchant_id',
        'amount',
        'payment_date',
        'payment_method',
        'reference_number',
        'notes',
        'status'
    ];

    protected $casts = [
        'payment_date' => 'date'
    ];

    public function merchant()
    {
        return $this->belongsTo(User::class, 'merchant_id');
    }
}

class Expense extends Model
{
    use HasFactory;

    protected $fillable = [
        'expense_number',
        'title',
        'description',
        'amount',
        'expense_date',
        'category',
        'approved_by',
        'status',
        'receipt'
    ];

    protected $casts = [
        'expense_date' => 'date'
    ];

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }
}

class CompanyAccount extends Model
{
    use HasFactory;

    protected $fillable = [
        'account_code',
        'account_name',
        'account_type',
        'account_category',
        'balance',
        'is_active',
        'description'
    ];

    protected $casts = [
        'is_active' => 'boolean'
    ];

    public function journalEntryLines()
    {
        return $this->hasMany(JournalEntryLine::class, 'account_id');
    }
}

class JournalEntry extends Model
{
    use HasFactory;

    protected $fillable = [
        'entry_number',
        'entry_date',
        'reference',
        'description',
        'total_debit',
        'total_credit',
        'status',
        'posted_by',
        'posted_at'
    ];

    protected $casts = [
        'entry_date' => 'date',
        'posted_at' => 'datetime'
    ];

    public function lines()
    {
        return $this->hasMany(JournalEntryLine::class, 'journal_entry_id');
    }

    public function poster()
    {
        return $this->belongsTo(User::class, 'posted_by');
    }
}

class JournalEntryLine extends Model
{
    use HasFactory;

    protected $fillable = [
        'journal_entry_id',
        'account_id',
        'type',
        'amount',
        'description'
    ];

    public function journalEntry()
    {
        return $this->belongsTo(JournalEntry::class, 'journal_entry_id');
    }

    public function account()
    {
        return $this->belongsTo(CompanyAccount::class, 'account_id');
    }
}
