@extends('layouts.admin')

@section('title', 'Chart of Accounts')

@section('content')
<div class="p-6 lg:p-8" x-data="{ showModal: false }" @open-modal.window="showModal = true">
    @if(session('success'))
        <div class="mb-4 bg-green-50 border border-green-200 text-green-800 px-4 py-3 rounded">
            <span class="text-sm">{{ session('success') }}</span>
        </div>
    @endif
    <!-- Page Header -->
    <div class="flex justify-between items-center mb-6">
        <div>
            <h1 class="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-800">Chart of Accounts</h1>
            <p class="text-sm text-gray-600">Manage all financial accounts for your company.</p>
        </div>
        <div class="flex items-center gap-2 sm:gap-3">
            <a href="{{ route('finance.accounting.chart-of-accounts.export', request()->only(['q','type','category','status'])) }}" class="bg-white border border-gray-300 text-gray-700 px-3 py-2 rounded-md hover:bg-gray-50">Export</a>
            <button @click="showModal = true" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Add New Account</button>
        </div>
    </div>

    <!-- Filters -->
    <form method="GET" action="{{ route('finance.accounting.chart-of-accounts') }}" class="mb-4 grid grid-cols-1 md:grid-cols-5 gap-2 sm:gap-3">
        <input type="text" name="q" value="{{ request('q') }}" placeholder="Search name/code..." class="border rounded-md px-3 py-2">
        <select name="type" class="border rounded-md px-3 py-2">
            <option value="">All Types</option>
            @foreach(['asset','liability','equity','revenue','expense'] as $t)
                <option value="{{ $t }}" {{ request('type')===$t ? 'selected' : '' }}>{{ ucfirst($t) }}</option>
            @endforeach
        </select>
        <select name="category" class="border rounded-md px-3 py-2">
            <option value="">All Categories</option>
            @foreach(['current_asset','fixed_asset','current_liability','long_term_liability','equity','revenue','cost_of_goods_sold','operating_expense','other'] as $c)
                <option value="{{ $c }}" {{ request('category')===$c ? 'selected' : '' }}>{{ ucwords(str_replace('_',' ', $c)) }}</option>
            @endforeach
        </select>
        <select name="status" class="border rounded-md px-3 py-2">
            <option value="">Any Status</option>
            <option value="active" {{ request('status')==='active' ? 'selected' : '' }}>Active</option>
            <option value="inactive" {{ request('status')==='inactive' ? 'selected' : '' }}>Inactive</option>
        </select>
        <button type="submit" class="bg-gray-800 text-white px-3 py-2 rounded-md">Apply Filters</button>
    </form>

    @if($errors->any() && old('_form') === 'create_account')
        <div class="mb-4 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded">
            <ul class="list-disc ml-4 text-sm">
                @foreach($errors->all() as $e)
                    <li>{{ $e }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <!-- Accounts Table -->
    <div class="bg-white rounded-lg shadow-md">
        <div class="p-4 border-b flex items-center justify-between">
            <h3 class="text-lg font-semibold">All Accounts</h3>
            <div class="text-sm text-gray-500">Total: {{ $accounts->total() }}</div>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Code</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Account Name</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Type</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Category</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Balance</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($accounts as $account)
                        <tr>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">{{ $account->account_code }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs font-medium text-gray-900">{{ $account->account_name }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">{{ $account->account_type }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">{{ $account->account_category }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">Rs. {{ number_format($account->balance ?? 0, 2) }}</td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                @if($account->is_active)
                                    <span class="px-2 inline-flex text-[10px] leading-4 font-semibold rounded-full bg-green-100 text-green-800">Active</span>
                                @else
                                    <span class="px-2 inline-flex text-[10px] leading-4 font-semibold rounded-full bg-red-100 text-red-800">Inactive</span>
                                @endif
                            </td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs font-medium">
                                <a href="{{ route('finance.accounting.chart-of-accounts.edit', $account) }}" class="text-indigo-600 hover:text-indigo-900">Edit</a>
                                <form action="{{ route('finance.accounting.chart-of-accounts.destroy', $account) }}" method="POST" class="inline" onsubmit="return confirm('Are you sure?')">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="text-red-600 hover:text-red-900 ml-2">Delete</button>
                                </form>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="text-center py-10 text-gray-500">No accounts found.</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        <div class="p-4 border-t">
            {{ $accounts->links() }}
        </div>
    </div>
    <!-- Add Account Modal -->
<div x-show="showModal" @click.away="showModal = false" class="fixed inset-0 bg-gray-600 bg-opacity-50 h-full w-full z-50" x-cloak>
    <div class="relative top-10 mx-auto p-5 border w-full max-w-lg shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold">Add New Account</h3>
            <button @click="showModal = false" class="text-gray-400 hover:text-gray-500">&times;</button>
        </div>
        <form action="{{ route('finance.accounting.chart-of-accounts.store') }}" method="POST">
            @csrf
            <input type="hidden" name="_form" value="create_account">
            <div>
                    <div class="mb-4">
                        <label for="account_name" class="block text-xs sm:text-sm font-medium text-gray-700">Account Name</label>
                        <input type="text" name="account_name" class="mt-1 form-input" value="{{ old('account_name') }}" required>
                    </div>
                    <div class="mb-4">
                        <label for="account_code" class="block text-xs sm:text-sm font-medium text-gray-700">Account Code</label>
                        <input type="text" name="account_code" class="mt-1 form-input" value="{{ old('account_code') }}" required>
                    </div>
                    <div class="mb-4">
                        <label for="account_type" class="block text-xs sm:text-sm font-medium text-gray-700">Account Type</label>
                        <select name="account_type" class="mt-1 form-select" required>
                            @foreach(['asset','liability','equity','revenue','expense'] as $t)
                                <option value="{{ $t }}" {{ old('account_type')===$t ? 'selected' : '' }}>{{ ucfirst($t) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-4">
                        <label for="account_category" class="block text-xs sm:text-sm font-medium text-gray-700">Account Category</label>
                        <select name="account_category" class="mt-1 form-select" required>
                            @foreach(['current_asset','fixed_asset','current_liability','long_term_liability','equity','revenue','cost_of_goods_sold','operating_expense','other'] as $c)
                                <option value="{{ $c }}" {{ old('account_category')===$c ? 'selected' : '' }}>{{ ucwords(str_replace('_',' ', $c)) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-4">
                        <label for="description" class="block text-xs sm:text-sm font-medium text-gray-700">Description</label>
                        <textarea name="description" class="mt-1 form-input">{{ old('description') }}</textarea>
                    </div>
                    <label class="inline-flex items-center text-sm"><input type="checkbox" name="is_active" value="1" class="mr-2" {{ old('is_active', 1) ? 'checked' : '' }}>Active</label>
                </div>
                <div class="flex justify-end space-x-2 mt-6">
                    <button type="button" @click="showModal = false" class="bg-gray-300 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-400">Close</button>
                    <button type="submit" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Save Account</button>
                </div>
            </form>
        </div>
    </div>
</div>


</div>
@endsection

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function () {
        @if($errors->any() && old('_form') === 'create_account')
            window.dispatchEvent(new CustomEvent('open-modal'));
        @endif
    });
</script>
@endpush



