@extends('layouts.admin')

@section('title', 'Bank Deposits - Finance')

@section('content')
<div class="container mx-auto px-2 sm:px-4 lg:px-6 py-3 sm:py-4 lg:py-6">
    <!-- Header with Date Filter & Export -->
    <div class="mb-4 sm:mb-6">
        <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
            <div>
                <h1 class="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-900">Bank Deposits Report</h1>
                <p class="text-xs sm:text-sm text-gray-600 mt-1">COD deposits to bank accounts on <span id="reportDate">{{ \Carbon\Carbon::parse($date)->format('F jS, Y') }}</span></p>
            </div>
            <div class="flex flex-col sm:flex-row gap-2 sm:gap-3">
                <div class="relative">
                    <input type="date" id="dateFilter" value="{{ $date }}" 
                           class="pl-10 pr-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-2 focus:ring-blue-500"
                           onchange="updateReport(this.value)">
                    <svg class="w-5 h-5 text-gray-400 absolute left-3 top-2.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                    </svg>
                </div>
                <button onclick="exportReport()" class="inline-flex items-center px-4 py-2 bg-white border border-gray-300 rounded-md hover:bg-gray-50 text-sm font-medium">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                    </svg>
                    Export Report
                </button>
                <button onclick="openDepositModal()" class="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
                    </svg>
                    New Deposit
                </button>
            </div>
        </div>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border border-green-400 text-green-700 px-3 sm:px-4 py-2 sm:py-3 rounded relative mb-4 text-xs sm:text-sm">
            {{ session('success') }}
        </div>
    @endif

    <!-- Stats Cards -->
    <div class="grid grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4 mb-4 sm:mb-6">
        <div class="bg-white rounded-lg shadow-md p-3 sm:p-4">
            <p class="text-[10px] sm:text-xs text-gray-600">Today's Deposits</p>
            <p class="text-lg sm:text-2xl font-bold text-gray-900">Rs. {{ number_format($stats['today_deposits']) }}</p>
        </div>
        <div class="bg-white rounded-lg shadow-md p-3 sm:p-4">
            <p class="text-[10px] sm:text-xs text-gray-600">Pending</p>
            <p class="text-lg sm:text-2xl font-bold text-yellow-600">Rs. {{ number_format($stats['pending']) }}</p>
        </div>
        <div class="bg-white rounded-lg shadow-md p-3 sm:p-4">
            <p class="text-[10px] sm:text-xs text-gray-600">Verified</p>
            <p class="text-lg sm:text-2xl font-bold text-green-600">Rs. {{ number_format($stats['verified']) }}</p>
        </div>
        <div class="bg-white rounded-lg shadow-md p-3 sm:p-4">
            <p class="text-[10px] sm:text-xs text-gray-600">Total Count</p>
            <p class="text-lg sm:text-2xl font-bold text-gray-900">{{ $stats['total_count'] }}</p>
        </div>
    </div>

    <!-- Deposits Table -->
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-3 sm:px-6 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Reference</th>
                        <th class="px-3 sm:px-6 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Bank</th>
                        <th class="px-3 sm:px-6 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                        <th class="px-3 sm:px-6 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                        <th class="px-3 sm:px-6 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($deposits as $deposit)
                    <tr>
                        <td class="px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm font-medium text-blue-600">
                            {{ $deposit->reference_number }}
                        </td>
                        <td class="px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm">
                            Bank Deposit
                        </td>
                        <td class="px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm font-semibold">
                            Rs. {{ number_format($deposit->amount, 2) }}
                        </td>
                        <td class="px-3 sm:px-6 py-3 sm:py-4">
                            <span class="px-2 py-1 text-[10px] sm:text-xs font-semibold rounded-full 
                                @if($deposit->status === 'completed') bg-green-100 text-green-800
                                @elseif($deposit->status === 'pending') bg-yellow-100 text-yellow-800
                                @elseif($deposit->status === 'failed') bg-red-100 text-red-800
                                @else bg-gray-100 text-gray-800
                                @endif">
                                {{ ucfirst($deposit->status) }}
                            </span>
                        </td>
                        <td class="px-3 sm:px-6 py-3 sm:py-4">
                            <div class="flex space-x-1 sm:space-x-2">
                                <a href="{{ url('finance/cod.deposits.show', $deposit->id) }}" 
                                   class="text-blue-600 hover:text-blue-900" title="View">
                                    <svg class="h-4 w-4 sm:h-5 sm:w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                                    </svg>
                                </a>
                                @if($deposit->status === 'pending')
                                <button onclick="verifyDeposit({{ $deposit->id }})" 
                                       class="text-green-600 hover:text-green-900" title="Verify">
                                    <svg class="h-4 w-4 sm:h-5 sm:w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                                    </svg>
                                </button>
                                @endif
                                <a href="{{ url('finance/cod.deposits.edit', $deposit->id) }}" 
                                   class="text-gray-600 hover:text-gray-900" title="Edit">
                                    <svg class="h-4 w-4 sm:h-5 sm:w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                                    </svg>
                                </a>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="5" class="px-6 py-12 text-center text-gray-500">
                            <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                            <p class="mt-4">No deposits recorded for this date</p>
                            <p class="text-sm mt-2">Create your first bank deposit</p>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal -->
<div id="depositModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 z-50 overflow-y-auto">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-2xl shadow-lg rounded-md bg-white">
        <h3 class="text-lg font-bold mb-4">New Bank Deposit</h3>
        <form method="POST" action="{{ url('finance/cod.deposits.store') }}">
            @csrf
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium mb-1">Bank Name *</label>
                    <select name="bank_name" required class="w-full px-3 py-2 border rounded-md">
                        <option value="">Select Bank</option>
                        <option value="HBL">HBL - Habib Bank Limited</option>
                        <option value="UBL">UBL - United Bank Limited</option>
                        <option value="MCB">MCB - Muslim Commercial Bank</option>
                        <option value="Allied Bank">Allied Bank</option>
                        <option value="Meezan Bank">Meezan Bank</option>
                        <option value="Askari Bank">Askari Bank</option>
                        <option value="Bank Alfalah">Bank Alfalah</option>
                        <option value="Faysal Bank">Faysal Bank</option>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium mb-1">Branch</label>
                    <input type="text" name="branch" class="w-full border border-gray-300 rounded-md px-2 sm:px-3 py-1.5 sm:py-2 text-xs sm:text-sm focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Main Branch">
                </div>

                <div>
                    <label class="block text-sm font-medium mb-1">Amount (Rs.) *</label>
                    <input type="number" name="amount" step="0.01" min="0" required class="w-full border border-gray-300 rounded-md px-2 sm:px-3 py-1.5 sm:py-2 text-xs sm:text-sm focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="0.00">
                </div>

                <div>
                    <label class="block text-sm font-medium mb-1">Reference Number *</label>
                    <input type="text" name="reference" required class="w-full border border-gray-300 rounded-md px-2 sm:px-3 py-1.5 sm:py-2 text-xs sm:text-sm focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="DEP-2025-001">
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium mb-1">Deposit Date *</label>
                    <input type="date" name="deposit_date" required class="w-full border border-gray-300 rounded-md px-2 sm:px-3 py-1.5 sm:py-2 text-xs sm:text-sm focus:outline-none focus:ring-2 focus:ring-blue-500" value="{{ date('Y-m-d') }}">
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium mb-1">Notes</label>
                    <textarea name="notes" rows="3" class="w-full px-3 py-2 border rounded-md" placeholder="Optional notes about this deposit"></textarea>
                </div>
            </div>

            <div class="mt-6 flex justify-end gap-3">
                <button type="button" onclick="closeDepositModal()" class="px-4 py-2 border rounded-md text-gray-700 hover:bg-gray-50">
                    Cancel
                </button>
                <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                    Save Deposit
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openDepositModal() { document.getElementById('depositModal').classList.remove('hidden'); }
function closeDepositModal() { document.getElementById('depositModal').classList.add('hidden'); }

function verifyDeposit(id) {
    if(confirm('Are you sure you want to verify this deposit? This will mark it as completed.')) {
        // Send AJAX request to verify the deposit
        fetch(`/admin/finance/cod/deposits/${id}/verify`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if(data.success) {
                alert('Deposit verified successfully!');
                location.reload(); // Refresh the page to show updated status
            } else {
                alert('Error verifying deposit: ' + data.message);
            }
        })
        .catch(error => {
            alert('Error verifying deposit');
            console.error('Error:', error);
        });
    }
}

function viewDeposit(id) {
    window.location.href = `/admin/finance/cod/deposits/${id}`;
}

function editDeposit(id) {
    window.location.href = `/admin/finance/cod/deposits/${id}/edit`;
}

function updateReport(date) {
    const dateObj = new Date(date);
    const options = { year: 'numeric', month: 'long', day: 'numeric' };
    document.getElementById('reportDate').textContent = dateObj.toLocaleDateString('en-US', options);
    window.location.href = `?date=${date}`;
}

function exportReport() {
    const date = document.getElementById('dateFilter').value;
    window.location.href = `/admin/finance/cod/deposits/export?date=${date}`;
}
</script>
@endsection

