@extends('layouts.admin')

@section('title', 'Company Expenses')

@section('content')
<div class="p-6 lg:p-8" x-data="{ showModal: false }" @open-modal.window="showModal = true">
    <!-- Page Header -->
    <div class="flex justify-between items-center mb-6">
        <div>
            <h1 class="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-800">Company Expenses</h1>
            <p class="text-sm text-gray-600">Track and manage all company-wide expenses.</p>
        </div>
        <button @click="showModal = true" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Add New Expense</button>
    </div>

    <!-- Filters -->
    <form method="GET" action="{{ route('finance.expenses.company') }}" class="bg-white rounded-lg shadow-md p-4 mb-6 grid grid-cols-1 md:grid-cols-4 gap-4">
        <input type="date" name="start_date" value="{{ request('start_date') }}" class="form-input">
        <input type="date" name="end_date" value="{{ request('end_date') }}" class="form-input">
        <select name="expense_category" class="form-select">
            <option value="">All Categories</option>
            @foreach(['office_supplies', 'utilities', 'rent', 'salary', 'marketing', 'travel', 'maintenance', 'other'] as $cat)
                <option value="{{ $cat }}" {{ request('expense_category') == $cat ? 'selected' : '' }}>{{ ucwords(str_replace('_', ' ', $cat)) }}</option>
            @endforeach
        </select>
        <button type="submit" class="bg-gray-800 text-white px-4 py-2 rounded-md">Filter</button>
    </form>

    <!-- Expenses Table -->
    <div class="bg-white rounded-lg shadow-md">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Date</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Title</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Category</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Amount</th>
                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Status</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($expenses as $expense)
                        <tr>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs">{{ $expense->expense_date->format('M d, Y') }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs font-medium">{{ $expense->title }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs">{{ ucwords(str_replace('_', ' ', $expense->expense_category)) }}</td>
                            <td class="px-6 py-4 text-right text-sm">{{ number_format($expense->amount, 2) }}</td>
                            <td class="px-6 py-4 text-center text-sm">
                                @php
                                    $statusColor = match($expense->status) {
                                        'approved' => ['bg' => 'bg-green-100', 'text' => 'text-green-800'],
                                        'paid' => ['bg' => 'bg-blue-100', 'text' => 'text-blue-800'],
                                        'rejected' => ['bg' => 'bg-red-100', 'text' => 'text-red-800'],
                                        default => ['bg' => 'bg-yellow-100', 'text' => 'text-yellow-800']
                                    };
                                @endphp
                                <span class="px-2 inline-flex text-[10px] leading-4 font-semibold rounded-full {{ $statusColor['bg'] }} {{ $statusColor['text'] }}">
                                    {{ ucfirst($expense->status) }}
                                </span>
                            </td>
                            <td class="px-6 py-4 text-right text-sm font-medium">
                                <a href="{{ route('finance.expenses.edit', $expense) }}" class="text-indigo-600 hover:text-indigo-900 mr-3">Edit</a>
                                <form action="{{ route('finance.expenses.destroy', $expense) }}" method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this expense?')">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="text-red-600 hover:text-red-900">Delete</button>
                                </form>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="text-center py-10 text-gray-500">No expenses found.</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        <div class="p-4 border-t">
            {{ $expenses->links() }}
        </div>
    </div>

    <!-- Add Expense Modal -->
    <div x-show="showModal" @keydown.escape.window="showModal = false" @click.away="showModal = false" class="fixed inset-0 bg-gray-600 bg-opacity-50 h-full w-full z-50" x-cloak>
        <div class="relative top-10 mx-auto p-5 border w-full max-w-xl shadow-xl rounded-lg bg-white" role="dialog" aria-modal="true" aria-labelledby="add-expense-title">
            <form action="{{ route('finance.expenses.store') }}" method="POST" enctype="multipart/form-data" novalidate>
                @csrf
                <h3 id="add-expense-title" class="text-lg font-semibold mb-4">Add New Expense</h3>
                @if ($errors->any())
                    <div class="mb-4 rounded-md bg-red-50 p-3">
                        <div class="text-red-700 text-sm font-semibold mb-1">Please fix the following:</div>
                        <ul class="list-disc list-inside text-xs text-red-700">
                            @foreach ($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                @endif
                <div class="space-y-4">
                    <div>
                        <label for="title" class="block text-xs sm:text-sm font-medium text-gray-700">Title</label>
                        <input id="title" type="text" name="title" value="{{ old('title') }}" class="mt-1 form-input placeholder-gray-400" placeholder="e.g., Printer ink" autocomplete="off" autofocus required>
                    </div>
                    <div>
                        <label for="expense_category" class="block text-xs sm:text-sm font-medium text-gray-700">Category</label>
                        <select id="expense_category" name="expense_category" class="mt-1 form-select" required>
                            @foreach(['office_supplies', 'utilities', 'rent', 'salary', 'marketing', 'travel', 'maintenance', 'other'] as $cat)
                                <option value="{{ $cat }}">{{ ucwords(str_replace('_', ' ', $cat)) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div>
                        <label for="amount" class="block text-xs sm:text-sm font-medium text-gray-700">Amount</label>
                        <input id="amount" type="number" inputmode="decimal" name="amount" value="{{ old('amount') }}" step="0.01" class="mt-1 form-input placeholder-gray-400" placeholder="e.g., 1200" required>
                    </div>
                    <div>
                        <label for="expense_date" class="block text-xs sm:text-sm font-medium text-gray-700">Expense Date</label>
                        <input id="expense_date" type="date" name="expense_date" value="{{ old('expense_date', now()->format('Y-m-d')) }}" class="mt-1 form-input" required>
                    </div>
                    <div>
                        <label for="description" class="block text-xs sm:text-sm font-medium text-gray-700">Description</label>
                        <textarea id="description" name="description" rows="3" class="mt-1 form-input placeholder-gray-400" placeholder="Optional notes">{{ old('description') }}</textarea>
                    </div>
                    <div>
                        <label for="receipt" class="block text-xs sm:text-sm font-medium text-gray-700">Receipt (optional)</label>
                        <input id="receipt" type="file" name="receipt" class="mt-1 form-input" accept="image/*,.pdf">
                        <p class="mt-1 text-[11px] text-gray-500">Accepted: images or PDF. Max size 5MB.</p>
                    </div>
                </div>
                <div class="flex justify-end space-x-2 mt-6">
                    <button type="button" @click="showModal = false" class="bg-gray-300 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-400">Close</button>
                    <button type="submit" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Save Expense</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    [x-cloak] { display: none !important; }
</style>
@endpush

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function () {
        @if($errors->any())
            window.dispatchEvent(new CustomEvent('open-modal'));
        @endif
    });
</script>
@endpush



