@extends('layouts.admin')

@section('title', 'Company Expenses')

@section('content')
<div class="p-6 lg:p-8" x-data="{ showModal: false }">
    <!-- Page Header -->
    <div class="flex justify-between items-center mb-6">
        <div>
            <h1 class="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-800">Company Expenses</h1>
            <p class="text-sm text-gray-600">Track and manage all company-level expenses.</p>
        </div>
        <div>
            <button @click="showModal = true" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Add Expense</button>
        </div>
    </div>

    <!-- Expenses Table -->
    <div class="bg-white rounded-lg shadow-md">
        <div class="p-4 border-b flex justify-between items-center">
            <h3 class="text-lg font-semibold">All Company Expenses</h3>
            <form method="GET" class="flex items-center space-x-2">
                <input type="date" name="start_date" class="form-input" value="{{ request('start_date') }}" placeholder="Start Date">
                <input type="date" name="end_date" class="form-input" value="{{ request('end_date') }}" placeholder="End Date">
                <button type="submit" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Filter</button>
            </form>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Expense #</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Title</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Category</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Date</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-4 py-2 text-left text-[11px] font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($expenses as $expense)
                        <tr>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">{{ $expense->expense_number }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs font-medium text-gray-900">{{ $expense->title }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">{{ ucfirst($expense->expense_category) }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">Rs. {{ number_format($expense->amount, 2) }}</td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs text-gray-500">{{ $expense->expense_date->format('M d, Y') }}</td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 inline-flex text-[10px] leading-4 font-semibold rounded-full 
                                    @if($expense->status == 'approved') bg-green-100 text-green-800
                                    @elseif($expense->status == 'pending') bg-yellow-100 text-yellow-800
                                    @else bg-red-100 text-red-800 @endif">
                                    {{ ucfirst($expense->status) }}
                                </span>
                            </td>
                            <td class="px-4 py-2.5 whitespace-nowrap text-xs font-medium">
                                <a href="{{ route('finance.expenses.edit', $expense) }}" class="text-indigo-600 hover:text-indigo-900 mr-3">Edit</a>
                                <form action="{{ route('finance.expenses.destroy', $expense) }}" method="POST" class="inline" onsubmit="return confirm('Are you sure?')">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="text-red-600 hover:text-red-900">Delete</button>
                                </form>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="text-center py-10 text-gray-500">No expenses found.</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        <div class="p-4">
            {{ $expenses->links() }}
        </div>
    </div>

    <!-- Add Expense Modal -->
    <div x-show="showModal" @keydown.escape.window="showModal = false" @click.away="showModal = false" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50" x-cloak>
        <div class="relative top-20 mx-auto p-5 border w-full max-w-xl shadow-xl rounded-lg bg-white" role="dialog" aria-modal="true" aria-labelledby="add-expense-title">
            <div class="flex justify-between items-center mb-4">
                <h3 id="add-expense-title" class="text-lg font-semibold">Add New Expense</h3>
                <button @click="showModal = false" class="text-gray-500 hover:text-gray-700" aria-label="Close add expense modal">
                    <svg class="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                    </svg>
                </button>
            </div>
            <form action="{{ route('finance.expenses.store') }}" method="POST" enctype="multipart/form-data" novalidate>
                @csrf
                <div class="space-y-4">
                    <div class="mb-4">
                        <label for="title" class="block text-xs sm:text-sm font-medium text-gray-700">Title</label>
                        <input id="title" type="text" name="title" value="{{ old('title') }}" autocomplete="off" autofocus class="mt-1 block w-full rounded-md border-gray-300 shadow-sm placeholder-gray-400" placeholder="e.g., Office chair purchase" required>
                        @error('title')
                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                    <div class="mb-4">
                        <label for="expense_category" class="block text-xs sm:text-sm font-medium text-gray-700">Category</label>
                        <select id="expense_category" name="expense_category" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" required>
                            <option value="office_supplies" {{ old('expense_category')=='office_supplies' ? 'selected' : '' }}>Office Supplies</option>
                            <option value="utilities" {{ old('expense_category')=='utilities' ? 'selected' : '' }}>Utilities</option>
                            <option value="rent" {{ old('expense_category')=='rent' ? 'selected' : '' }}>Rent</option>
                            <option value="salary" {{ old('expense_category')=='salary' ? 'selected' : '' }}>Salary</option>
                            <option value="marketing" {{ old('expense_category')=='marketing' ? 'selected' : '' }}>Marketing</option>
                            <option value="travel" {{ old('expense_category')=='travel' ? 'selected' : '' }}>Travel</option>
                            <option value="maintenance" {{ old('expense_category')=='maintenance' ? 'selected' : '' }}>Maintenance</option>
                            <option value="other" {{ old('expense_category')=='other' ? 'selected' : '' }}>Other</option>
                        </select>
                        @error('expense_category')
                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                    <div class="mb-4">
                        <label for="amount" class="block text-xs sm:text-sm font-medium text-gray-700">Amount</label>
                        <input id="amount" type="number" inputmode="decimal" name="amount" value="{{ old('amount') }}" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm placeholder-gray-400" step="0.01" placeholder="e.g., 5000" required>
                        @error('amount')
                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                    <div class="mb-4">
                        <label for="expense_date" class="block text-xs sm:text-sm font-medium text-gray-700">Expense Date</label>
                        <input id="expense_date" type="date" name="expense_date" value="{{ old('expense_date', now()->format('Y-m-d')) }}" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" required>
                        @error('expense_date')
                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                    <div class="mb-4">
                        <label for="description" class="block text-xs sm:text-sm font-medium text-gray-700">Description</label>
                        <textarea id="description" name="description" rows="3" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm placeholder-gray-400" placeholder="Optional notes about this expense">{{ old('description') }}</textarea>
                        <p class="mt-1 text-[11px] text-gray-500">Add any details that help identify or justify the expense.</p>
                        @error('description')
                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                    <div class="mb-2">
                        <label for="receipt" class="block text-xs sm:text-sm font-medium text-gray-700">Receipt (optional)</label>
                        <input id="receipt" type="file" name="receipt" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" accept="image/*,.pdf">
                        <p class="mt-1 text-[11px] text-gray-500">Accepted: images or PDF. Max size 5MB.</p>
                        @error('receipt')
                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>
                <div class="flex justify-end space-x-2 mt-6">
                    <button type="button" @click="showModal = false" class="bg-gray-300 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-400">Close</button>
                    <button type="submit" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Save Expense</button>
                </div>
            </form>
        </div>
    </div>
</div><!-- End Alpine.js scope -->
@endsection

@push('styles')
<style>
    [x-cloak] { display: none !important; }
</style>
@endpush



