@extends('layouts.admin')

@section('title', 'Reports Dashboard')

@section('content')
<div class="container mx-auto px-2 sm:px-4 lg:px-6 py-3 sm:py-4 lg:py-6">
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-900">Reports Dashboard</h1>
            <p class="text-xs sm:text-sm text-gray-600 mt-1">Overview of finance and operations KPIs</p>
        </div>
        <div class="flex items-center gap-2 sm:gap-3">
            <a href="{{ url('finance/reports.financial') }}" class="text-xs px-3 py-1.5 rounded-md border border-gray-300 hover:bg-gray-50">Financial</a>
            <a href="{{ url('finance/reports.operational') }}" class="text-xs px-3 py-1.5 rounded-md border border-gray-300 hover:bg-gray-50">Operational</a>
            <a href="{{ url('finance/reports.custom') }}" class="text-xs px-3 py-1.5 rounded-md border border-gray-300 hover:bg-gray-50">Custom</a>
            <a href="{{ url('finance/reports.export') }}" class="text-xs px-3 py-1.5 rounded-md border border-gray-300 hover:bg-gray-50">Export</a>
        </div>
    </div>

    <!-- Date Range Filters -->
    <div class="bg-white rounded-lg border border-gray-200 p-4 mb-6">
        <form id="filters-form" class="flex flex-col sm:flex-row items-start sm:items-end gap-3">
            <div>
                <label for="start-date" class="block text-xs text-gray-600">Start date</label>
                <input type="date" id="start-date" class="mt-1 text-sm border border-gray-300 rounded-md px-2 py-1" />
            </div>
            <div>
                <label for="end-date" class="block text-xs text-gray-600">End date</label>
                <input type="date" id="end-date" class="mt-1 text-sm border border-gray-300 rounded-md px-2 py-1" />
            </div>
            <div class="flex items-center gap-2 sm:gap-3">
                <button type="submit" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">Apply</button>
                <button type="button" id="reset-btn" class="mt-6 sm:mt-0 text-sm px-3 py-1.5 rounded-md border border-gray-300 hover:bg-gray-50">Reset</button>
            </div>
        </form>
    </div>

    <!-- Summary Cards -->
    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4 mb-6" id="summary-cards">
        <div class="bg-white rounded-lg border border-gray-200 p-4">
            <p class="text-xs text-gray-500">Total Invoices</p>
            <p class="mt-1 text-xl font-semibold text-gray-900" id="metric-total-invoices">—</p>
        </div>
        <div class="bg-white rounded-lg border border-gray-200 p-4">
            <p class="text-xs text-gray-500">Outstanding Amount</p>
            <p class="mt-1 text-xl font-semibold text-gray-900" id="metric-outstanding">—</p>
        </div>
        <div class="bg-white rounded-lg border border-gray-200 p-4">
            <p class="text-xs text-gray-500">Total Payments</p>
            <p class="mt-1 text-xl font-semibold text-gray-900" id="metric-total-payments">—</p>
        </div>
        <div class="bg-white rounded-lg border border-gray-200 p-4">
            <p class="text-xs text-gray-500">Approved Expenses</p>
            <p class="mt-1 text-xl font-semibold text-gray-900" id="metric-approved-expenses">—</p>
        </div>
        <div class="bg-white rounded-lg border border-gray-200 p-4">
            <p class="text-xs text-gray-500">Delivered Parcels</p>
            <p class="mt-1 text-xl font-semibold text-gray-900" id="metric-delivered-parcels">—</p>
        </div>
    </div>

    <!-- Deliveries Trend (Last 7 Days) -->
    <div class="bg-white rounded-lg shadow-md p-3 sm:p-4 lg:p-6">
        <div class="flex items-center justify-between mb-3">
            <div>
                <h2 class="text-base font-semibold text-gray-900">Deliveries - Last 7 Days</h2>
                <p class="text-xs text-gray-500">Counts based on parcel updates</p>
            </div>
            <div id="chart-legend" class="text-xs text-gray-500"></div>
        </div>
        <div id="deliveries-chart" class="w-full h-48 relative">
            <div id="chart-loading" class="absolute inset-0 flex items-center justify-center text-gray-500 text-sm">Loading chart...</div>
            <svg id="chart-svg" class="w-full h-full"></svg>
        </div>
    </div>

    <!-- Notes -->
    <div class="mt-6 text-xs text-gray-500">
        <p>Data updates automatically from the dashboard endpoint.</p>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    const baseUrl = "{{ url('finance/reports.dashboard.data') }}";

    const elTotalInvoices = document.getElementById('metric-total-invoices');
    const elOutstanding = document.getElementById('metric-outstanding');
    const elTotalPayments = document.getElementById('metric-total-payments');
    const elApprovedExpenses = document.getElementById('metric-approved-expenses');
    const elDeliveredParcels = document.getElementById('metric-delivered-parcels');

    const chartSvg = document.getElementById('chart-svg');
    const chartLoading = document.getElementById('chart-loading');
    const chartLegend = document.getElementById('chart-legend');

    function formatCurrency(value) {
        try {
            return new Intl.NumberFormat(undefined, { style: 'currency', currency: 'PKR', maximumFractionDigits: 0 }).format(value);
        } catch (e) {
            return '₨ ' + Number(value).toLocaleString();
        }
    }

    function renderChart(labels, series) {
        chartLoading.style.display = 'none';
        // Clear previous
        while (chartSvg.firstChild) chartSvg.removeChild(chartSvg.firstChild);

        const width = chartSvg.clientWidth || chartSvg.parentElement.clientWidth;
        const height = chartSvg.clientHeight || 192;
        chartSvg.setAttribute('viewBox', `0 0 ${width} ${height}`);

        const padding = { top: 10, right: 10, bottom: 24, left: 24 };
        const chartW = width - padding.left - padding.right;
        const chartH = height - padding.top - padding.bottom;

        const maxVal = Math.max(1, ...series);
        const barW = chartW / series.length * 0.6;
        const gap = chartW / series.length * 0.4;

        series.forEach((val, i) => {
            const x = padding.left + i * (barW + gap) + gap * 0.5;
            const h = Math.round((val / maxVal) * chartH);
            const y = padding.top + (chartH - h);

            const rect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
            rect.setAttribute('x', x);
            rect.setAttribute('y', y);
            rect.setAttribute('width', barW);
            rect.setAttribute('height', h);
            rect.setAttribute('fill', '#3b82f6');
            rect.setAttribute('rx', 3);
            chartSvg.appendChild(rect);

            // Label
            const label = document.createElementNS('http://www.w3.org/2000/svg', 'text');
            label.setAttribute('x', x + barW / 2);
            label.setAttribute('y', height - 5);
            label.setAttribute('text-anchor', 'middle');
            label.setAttribute('font-size', '10');
            label.setAttribute('fill', '#6b7280');
            label.textContent = labels[i].slice(5); // show MM-DD
            chartSvg.appendChild(label);
        });

        chartLegend.textContent = 'Delivered parcels per day';
    }

    const startInput = document.getElementById('start-date');
    const endInput = document.getElementById('end-date');
    const form = document.getElementById('filters-form');
    const resetBtn = document.getElementById('reset-btn');

    function buildUrl() {
        const params = new URLSearchParams();
        if (startInput.value) params.set('start', startInput.value);
        if (endInput.value) params.set('end', endInput.value);
        return `${baseUrl}?${params.toString()}`;
    }

    function loadData() {
        chartLoading.style.display = 'flex';
        fetch(buildUrl())
            .then(r => r.json())
            .then(data => {
                const s = data.summary || {};
                elTotalInvoices.textContent = (s.total_invoices ?? '—').toLocaleString();
                elOutstanding.textContent = formatCurrency(s.outstanding_amount ?? 0);
                elTotalPayments.textContent = formatCurrency(s.total_payments ?? 0);
                elApprovedExpenses.textContent = formatCurrency(s.approved_expenses ?? 0);
                elDeliveredParcels.textContent = (s.delivered_parcels ?? '—').toLocaleString();

                renderChart(data.labels || [], data.series || []);
            })
            .catch(err => {
                console.error('Dashboard data error:', err);
                chartLoading.textContent = 'Failed to load data';
            });
    }

    form.addEventListener('submit', (e) => {
        e.preventDefault();
        loadData();
    });

    resetBtn.addEventListener('click', () => {
        startInput.value = '';
        endInput.value = '';
        loadData();
    });

    // Initial load
    loadData();
});
</script>
@endsection



