@extends('layouts.admin')

@section('title', 'Create Invoice')

@push('styles')
<style>
    body {
        background: #f0f2f5;
    }
    
    .invoice-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        padding: 30px 0;
        margin: -20px -15px 30px -15px;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }
    
    .invoice-card {
        background: white;
        border-radius: 20px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.08);
        border: none;
        overflow: hidden;
        transition: all 0.3s ease;
    }
    
    .invoice-card:hover {
        box-shadow: 0 15px 50px rgba(0,0,0,0.12);
        transform: translateY(-2px);
    }
    
    .form-control, .form-select {
        border-radius: 12px;
        border: 2px solid #e2e8f0;
        padding: 14px 18px;
        font-size: 14px;
        transition: all 0.3s ease;
        background: white;
    }
    
    .form-control:focus, .form-select:focus {
        border-color: #667eea;
        box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1);
        transform: translateY(-1px);
    }
    
    .section-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 20px 25px;
        border-radius: 15px 15px 0 0;
        margin: -24px -24px 25px -24px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }
    
    .item-row {
        background: linear-gradient(to right, #f8fafc, #ffffff);
        border-radius: 15px;
        padding: 20px;
        margin-bottom: 20px;
        border: 2px solid #e2e8f0;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    
    .item-row::before {
        content: '';
        position: absolute;
        left: 0;
        top: 0;
        height: 100%;
        width: 4px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        opacity: 0;
        transition: opacity 0.3s;
    }
    
    .item-row:hover::before {
        opacity: 1;
    }
    
    .item-row:hover {
        border-color: #667eea;
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.15);
        transform: translateX(5px);
    }
    
    .btn-add-item {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border: none;
        border-radius: 15px;
        padding: 15px 30px;
        color: white;
        font-weight: 600;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }
    
    .btn-add-item:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 30px rgba(102, 126, 234, 0.4);
        background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
    }
    
    .summary-card {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 20px;
        color: white;
        padding: 30px;
        box-shadow: 0 15px 40px rgba(102, 126, 234, 0.3);
        position: relative;
        overflow: hidden;
    }
    
    .summary-card::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -50%;
        width: 200%;
        height: 200%;
        background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
        animation: pulse 3s ease-in-out infinite;
    }
    
    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.05); }
    }
    
    .summary-row {
        display: flex;
        justify-content: space-between;
        padding: 12px 0;
        border-bottom: 1px solid rgba(255,255,255,0.2);
        position: relative;
        z-index: 1;
    }
    
    .summary-row:last-child {
        border-bottom: none;
        padding-top: 20px;
        margin-top: 15px;
        border-top: 2px solid rgba(255,255,255,0.4);
        font-size: 18px;
    }
    
    .item-number {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        width: 40px;
        height: 40px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        font-size: 16px;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        flex-shrink: 0;
    }
    
    .tax-discount-card {
        background: white;
        border-radius: 20px;
        box-shadow: 0 8px 30px rgba(0,0,0,0.08);
        border: 2px solid #f0f2f5;
        overflow: hidden;
    }
    
    .input-group-text {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        border-radius: 10px 0 0 10px;
    }
    
    .merchant-select {
        position: relative;
    }
    
    .merchant-select .form-select {
        padding-right: 45px;
    }
    
    .quick-actions {
        display: flex;
        gap: 10px;
        margin-bottom: 20px;
    }
    
    .quick-btn {
        flex: 1;
        padding: 12px;
        border-radius: 12px;
        border: 2px dashed #cbd5e0;
        background: white;
        color: #64748b;
        font-size: 13px;
        font-weight: 600;
        transition: all 0.3s;
        cursor: pointer;
    }
    
    .quick-btn:hover {
        border-color: #667eea;
        color: #667eea;
        background: #f8fafc;
    }
    
    .quick-btn i {
        display: block;
        font-size: 20px;
        margin-bottom: 5px;
    }
    
    .preview-badge {
        position: absolute;
        top: 20px;
        right: 20px;
        background: rgba(255,255,255,0.2);
        backdrop-filter: blur(10px);
        color: white;
        padding: 8px 20px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        letter-spacing: 1px;
    }
    
    .step-indicator {
        display: flex;
        justify-content: center;
        gap: 15px;
        margin-bottom: 30px;
    }
    
    .step {
        width: 12px;
        height: 12px;
        border-radius: 50%;
        background: #cbd5e0;
        transition: all 0.3s;
    }
    
    .step.active {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        width: 40px;
        border-radius: 10px;
    }
</style>
@endpush

@section('content')
<div class="container-fluid py-4">
    <!-- Professional Header -->
    <div class="invoice-header">
        <div class="container-fluid">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1 class="h2 mb-2 fw-bold text-white">
                        <i class="fas fa-file-invoice-dollar me-3"></i>
                        Create Professional Invoice
                    </h1>
                    <p class="text-white-50 mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        Generate detailed invoices with automatic calculations and professional formatting
                    </p>
                    <div class="step-indicator mt-3">
                        <div class="step active" data-step="1"></div>
                        <div class="step" data-step="2"></div>
                        <div class="step" data-step="3"></div>
                    </div>
                </div>
                <div class="col-md-4 text-end">
                    <div class="preview-badge">
                        <i class="fas fa-drafting-compass me-2"></i>DRAFT MODE
                    </div>
<a href="{{ route('admin.finance.settlements.invoices.index') }}" class="btn btn-light rounded-pill px-4 mt-3">
                        <i class="fas fa-arrow-left me-2"></i> Back to Invoices
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Database Stats Info -->
    @if(isset($stats))
    <div class="row mb-3">
        <div class="col-12">
            <div class="alert alert-info border-0 shadow-sm" style="background: linear-gradient(135deg, #e0f2fe 0%, #dbeafe 100%); border-radius: 15px;">
                <div class="d-flex align-items-center">
                    <i class="fas fa-database text-primary me-3" style="font-size: 24px;"></i>
                    <div class="flex-grow-1">
                        <strong class="text-primary">Database Status:</strong>
                        <span class="text-dark ms-2">
                            {{ $stats['total_merchants'] }} Merchants |
                            {{ $stats['pending_invoices'] }} Pending Invoices |
                            {{ $stats['draft_invoices'] }} Drafts
                        </span>
                    </div>
                    <span class="badge bg-success">
                        <i class="fas fa-check-circle me-1"></i>Live Data
                    </span>
                </div>
            </div>
        </div>
    </div>
    @endif

    <!-- Quick Actions -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="quick-actions">
                <button type="button" class="quick-btn" onclick="clearForm()">
                    <i class="fas fa-eraser"></i>
                    Clear Form
                </button>
                <button type="button" class="quick-btn" onclick="addMultipleItems()">
                    <i class="fas fa-layer-group"></i>
                    Add 3 Items
                </button>
                <button type="button" class="quick-btn" onclick="setDueDateDefault()">
                    <i class="fas fa-calendar-check"></i>
                    Set 30 Days Due
                </button>
                <button type="button" class="quick-btn" onclick="calculateTax()">
                    <i class="fas fa-percent"></i>
                    Auto Calculate Tax
                </button>
            </div>
        </div>
    </div>

<form action="{{ route('admin.finance.settlements.invoices.store') }}" method="POST" id="invoiceForm">
        @csrf
        
        <div class="row g-4">
            <!-- Left Column -->
            <div class="col-lg-8">
                <!-- Basic Information Card -->
                <div class="card invoice-card mb-4">
                    <div class="card-body p-4">
                        <div class="section-header">
                            <h5 class="mb-0 fw-bold"><i class="fas fa-info-circle me-2"></i>Invoice Information</h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label fw-semibold">
                                    <i class="fas fa-store text-primary me-2"></i>Merchant <span class="text-danger">*</span>
                                </label>
                                <select name="merchant_id" id="merchant_id" class="form-select @error('merchant_id') is-invalid @enderror" required>
                                    <option value="">-- Select Merchant --</option>
                                    @forelse($merchants as $merchant)
                                        <option value="{{ $merchant->id }}" {{ old('merchant_id') == $merchant->id ? 'selected' : '' }}>
                                            {{ $merchant->name }}
                                            @if($merchant->company_name)
                                                ({{ $merchant->company_name }})
                                            @endif
                                            - {{ $merchant->email }}
                                        </option>
                                    @empty
                                        <option value="" disabled>No merchants found. Please add merchants first.</option>
                                    @endforelse
                                </select>
                                @error('merchant_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                @if($merchants->count() > 0)
                                    <small class="text-muted"><i class="fas fa-info-circle me-1"></i>{{ $merchants->count() }} merchants available</small>
                                @else
                                    <small class="text-danger"><i class="fas fa-exclamation-triangle me-1"></i>No merchants found in the system</small>
                                @endif
                            </div>

                            <div class="col-md-3">
                                <label class="form-label fw-semibold">
                                    <i class="fas fa-calendar text-primary me-2"></i>Invoice Date <span class="text-danger">*</span>
                                </label>
                                <input type="date" name="invoice_date" id="invoice_date" class="form-control @error('invoice_date') is-invalid @enderror" value="{{ old('invoice_date', date('Y-m-d')) }}" required>
                                @error('invoice_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-3">
                                <label class="form-label fw-semibold">
                                    <i class="fas fa-calendar-alt text-danger me-2"></i>Due Date <span class="text-danger">*</span>
                                </label>
                                <input type="date" name="due_date" id="due_date" class="form-control @error('due_date') is-invalid @enderror" value="{{ old('due_date') }}" required>
                                @error('due_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Invoice Items Card -->
                <div class="card invoice-card mb-4">
                    <div class="card-body p-4">
                        <div class="section-header">
                            <h5 class="mb-0 fw-bold"><i class="fas fa-list-ul me-2"></i>Invoice Items</h5>
                        </div>

                        <div id="invoice-items">
                            <div class="invoice-item item-row">
                                <div class="d-flex align-items-start gap-3">
                                    <div class="item-number">1</div>
                                    <div class="flex-grow-1">
                                        <div class="row g-3">
                                            <div class="col-md-12">
                                                <label class="form-label fw-semibold small">Description <span class="text-danger">*</span></label>
                                                <input type="text" name="items[0][description]" class="form-control" placeholder="e.g., Delivery Service for Package #XYZ" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label fw-semibold small">Quantity <span class="text-danger">*</span></label>
                                                <input type="number" name="items[0][quantity]" class="form-control item-quantity" min="1" value="1" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label fw-semibold small">Unit Price (Rs.) <span class="text-danger">*</span></label>
                                                <input type="number" name="items[0][unit_price]" class="form-control item-price" step="0.01" min="0" value="0" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label fw-semibold small">Line Total</label>
                                                <input type="text" class="form-control item-total" readonly value="Rs. 0.00">
                                            </div>
                                        </div>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-outline-danger remove-item mt-4" disabled>
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>

                        <button type="button" class="btn btn-add-item w-100 mt-3" id="add-item">
                            <i class="fas fa-plus-circle me-2"></i> Add Another Item
                        </button>
                    </div>
                </div>

                <!-- Notes Card -->
                <div class="card invoice-card">
                    <div class="card-body p-4">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-sticky-note text-warning me-2"></i>Additional Notes
                        </label>
                        <textarea name="notes" id="notes" class="form-control" rows="4" placeholder="Enter any additional notes or terms...">{{ old('notes') }}</textarea>
                        <small class="text-muted">Optional: Add payment terms, thank you message, or any other relevant information</small>
                    </div>
                </div>
            </div>

            <!-- Right Column - Summary -->
            <div class="col-lg-4">
                <div class="position-sticky" style="top: 20px;">
                    <!-- Summary Card -->
                    <div class="summary-card mb-4">
                        <h5 class="fw-bold mb-4"><i class="fas fa-calculator me-2"></i>Invoice Summary</h5>
                        
                        <div class="summary-row">
                            <span>Subtotal:</span>
                            <span class="fw-bold" id="display-subtotal">Rs. 0.00</span>
                        </div>
                        
                        <div class="summary-row">
                            <span>Tax:</span>
                            <span class="fw-bold" id="display-tax">Rs. 0.00</span>
                        </div>
                        
                        <div class="summary-row">
                            <span>Discount:</span>
                            <span class="fw-bold" id="display-discount">Rs. 0.00</span>
                        </div>
                        
                        <div class="summary-row">
                            <span class="fs-5">Total Amount:</span>
                            <span class="fw-bold fs-4" id="display-total">Rs. 0.00</span>
                        </div>

                        <input type="hidden" name="subtotal" id="subtotal" value="0">
                    </div>

                    <!-- Tax & Discount Card -->
                    <div class="card invoice-card mb-4">
                        <div class="card-body p-4">
                            <h6 class="fw-bold mb-3 text-primary"><i class="fas fa-percent me-2"></i>Tax & Discounts</h6>
                            
                            <div class="mb-3">
                                <label class="form-label fw-semibold small">Tax Amount (Rs.)</label>
                                <div class="input-group">
                                    <span class="input-group-text"><i class="fas fa-receipt"></i></span>
                                    <input type="number" name="tax_amount" id="tax_amount" class="form-control" step="0.01" min="0" value="{{ old('tax_amount', 0) }}" placeholder="0.00">
                                </div>
                            </div>

                            <div class="mb-0">
                                <label class="form-label fw-semibold small">Discount Amount (Rs.)</label>
                                <div class="input-group">
                                    <span class="input-group-text"><i class="fas fa-tag"></i></span>
                                    <input type="number" name="discount_amount" id="discount_amount" class="form-control" step="0.01" min="0" value="{{ old('discount_amount', 0) }}" placeholder="0.00">
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-lg btn-primary rounded-pill">
                            <i class="fas fa-save me-2"></i> Create Invoice
                        </button>
<a href="{{ route('admin.finance.settlements.invoices.index') }}" class="btn btn-lg btn-outline-secondary rounded-pill">
                            <i class="fas fa-times me-2"></i> Cancel
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>

@push('scripts')
<script>
let itemIndex = 1;

// Add new item
document.getElementById('add-item').addEventListener('click', function() {
    itemIndex++;
    const container = document.getElementById('invoice-items');
    const newItem = document.createElement('div');
    newItem.className = 'invoice-item item-row';
    newItem.innerHTML = `
        <div class="d-flex align-items-start gap-3">
            <div class="item-number">${itemIndex}</div>
            <div class="flex-grow-1">
                <div class="row g-3">
                    <div class="col-md-12">
                        <label class="form-label fw-semibold small">Description <span class="text-danger">*</span></label>
                        <input type="text" name="items[${itemIndex - 1}][description]" class="form-control" placeholder="e.g., Delivery Service for Package #XYZ" required>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label fw-semibold small">Quantity <span class="text-danger">*</span></label>
                        <input type="number" name="items[${itemIndex - 1}][quantity]" class="form-control item-quantity" min="1" value="1" required>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label fw-semibold small">Unit Price (Rs.) <span class="text-danger">*</span></label>
                        <input type="number" name="items[${itemIndex - 1}][unit_price]" class="form-control item-price" step="0.01" min="0" value="0" required>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label fw-semibold small">Line Total</label>
                        <input type="text" class="form-control item-total" readonly value="Rs. 0.00">
                    </div>
                </div>
            </div>
            <button type="button" class="btn btn-sm btn-outline-danger remove-item mt-4">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    container.appendChild(newItem);
    updateItemNumbers();
    calculateTotal();
    updateRemoveButtons();
});

// Remove item
document.getElementById('invoice-items').addEventListener('click', function(e) {
    if (e.target.closest('.remove-item')) {
        e.target.closest('.invoice-item').remove();
        updateItemNumbers();
        calculateTotal();
        updateRemoveButtons();
    }
});

// Calculate on input
document.getElementById('invoice-items').addEventListener('input', function(e) {
    if (e.target.classList.contains('item-quantity') || e.target.classList.contains('item-price')) {
        const row = e.target.closest('.invoice-item');
        updateLineTotal(row);
        calculateTotal();
    }
});

document.getElementById('tax_amount').addEventListener('input', calculateTotal);
document.getElementById('discount_amount').addEventListener('input', calculateTotal);

function updateLineTotal(row) {
    const quantity = parseFloat(row.querySelector('.item-quantity').value) || 0;
    const price = parseFloat(row.querySelector('.item-price').value) || 0;
    const total = quantity * price;
    row.querySelector('.item-total').value = 'Rs. ' + total.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

function calculateTotal() {
    let subtotal = 0;
    document.querySelectorAll('.invoice-item').forEach(function(item) {
        const quantity = parseFloat(item.querySelector('.item-quantity').value) || 0;
        const price = parseFloat(item.querySelector('.item-price').value) || 0;
        subtotal += quantity * price;
        updateLineTotal(item);
    });

    const taxAmount = parseFloat(document.getElementById('tax_amount').value) || 0;
    const discountAmount = parseFloat(document.getElementById('discount_amount').value) || 0;
    const total = subtotal + taxAmount - discountAmount;

    // Format with commas
    const formatCurrency = (num) => 'Rs. ' + num.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");

    document.getElementById('subtotal').value = subtotal.toFixed(2);
    document.getElementById('display-subtotal').textContent = formatCurrency(subtotal);
    document.getElementById('display-tax').textContent = formatCurrency(taxAmount);
    document.getElementById('display-discount').textContent = formatCurrency(discountAmount);
    document.getElementById('display-total').textContent = formatCurrency(total);
}

function updateRemoveButtons() {
    const items = document.querySelectorAll('.invoice-item');
    items.forEach((item, index) => {
        const removeBtn = item.querySelector('.remove-item');
        removeBtn.disabled = items.length === 1;
    });
}

function updateItemNumbers() {
    document.querySelectorAll('.item-number').forEach((num, index) => {
        num.textContent = index + 1;
    });
}

// Initial calculation
calculateTotal();
updateRemoveButtons();

// Form validation with better UX
document.getElementById('invoiceForm').addEventListener('submit', function(e) {
    const merchant = document.getElementById('merchant_id').value;
    const invoiceDate = document.getElementById('invoice_date').value;
    const dueDate = document.getElementById('due_date').value;
    
    if (!merchant || !invoiceDate || !dueDate) {
        e.preventDefault();
        showNotification('Please fill in all required fields', 'error');
        return false;
    }
    
    const items = document.querySelectorAll('.invoice-item');
    if (items.length === 0) {
        e.preventDefault();
        showNotification('Please add at least one invoice item', 'error');
        return false;
    }
    
    // Show loading
    const submitBtn = this.querySelector('button[type="submit"]');
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Creating Invoice...';
    submitBtn.disabled = true;
});

// Add Multiple Items at Once
function addMultipleItems() {
    for (let i = 0; i < 3; i++) {
        document.getElementById('add-item').click();
    }
    showNotification('Added 3 new items', 'success');
}

// Set Due Date to 30 Days from Invoice Date
function setDueDateDefault() {
    const invoiceDate = document.getElementById('invoice_date').value;
    if (!invoiceDate) {
        showNotification('Please select invoice date first', 'warning');
        return;
    }
    
    const dueDate = new Date(invoiceDate);
    dueDate.setDate(dueDate.getDate() + 30);
    document.getElementById('due_date').value = dueDate.toISOString().split('T')[0];
    showNotification('Due date set to 30 days from invoice date', 'success');
}

// Auto Calculate Tax (18% GST)
function calculateTax() {
    const subtotal = parseFloat(document.getElementById('subtotal').value) || 0;
    if (subtotal === 0) {
        showNotification('Please add items first', 'warning');
        return;
    }
    
    const taxAmount = (subtotal * 0.18); // 18% GST
    document.getElementById('tax_amount').value = taxAmount.toFixed(2);
    calculateTotal();
    showNotification('Tax calculated at 18% GST', 'success');
}

// Clear Form
function clearForm() {
    if (confirm('Are you sure you want to clear all data?')) {
        document.getElementById('invoiceForm').reset();
        // Remove all items except first
        const items = document.querySelectorAll('.invoice-item');
        items.forEach((item, index) => {
            if (index > 0) item.remove();
        });
        updateItemNumbers();
        calculateTotal();
        updateRemoveButtons();
        showNotification('Form cleared successfully', 'info');
        updateStepIndicator(1);
    }
}


// Show Notification
function showNotification(message, type = 'info') {
    const colors = {
        success: '#10b981',
        error: '#ef4444',
        info: '#3b82f6',
        warning: '#f59e0b'
    };
    
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${colors[type]};
        color: white;
        padding: 15px 25px;
        border-radius: 10px;
        box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        z-index: 9999;
        font-weight: 600;
        animation: slideIn 0.3s ease;
    `;
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Update Step Indicator
function updateStepIndicator(step) {
    document.querySelectorAll('.step').forEach((el, index) => {
        if (index < step) {
            el.classList.add('active');
        } else {
            el.classList.remove('active');
        }
    });
}

// Auto-progress steps based on form completion
document.getElementById('merchant_id').addEventListener('change', function() {
    if (this.value) updateStepIndicator(2);
});

document.getElementById('invoice-items').addEventListener('input', function() {
    const hasItems = document.querySelectorAll('.invoice-item').length > 0;
    const firstItemFilled = document.querySelector('.item-quantity').value > 0;
    if (hasItems && firstItemFilled) updateStepIndicator(3);
});

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);

// Add keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl/Cmd + S to save
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        document.getElementById('invoiceForm').dispatchEvent(new Event('submit'));
    }
    // Ctrl/Cmd + K to add item
    if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
        e.preventDefault();
        document.getElementById('add-item').click();
    }
});

// Show keyboard shortcuts hint
console.log('%c💡 Keyboard Shortcuts', 'color: #667eea; font-size: 16px; font-weight: bold;');
console.log('Ctrl/Cmd + S: Save Invoice');
console.log('Ctrl/Cmd + K: Add New Item');
</script>
@endpush
@endsection
