@extends('layouts.admin')

@section('title', 'Contract Vault')

@section('content')
<div class="container mx-auto px-2 sm:px-4 lg:px-6 py-3 sm:py-4 lg:py-6">
    <div class="mb-4 sm:mb-6 flex justify-between items-center">
        <div>
            <h1 class="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-900">Merchant Contract Vault</h1>
            <p class="text-xs sm:text-sm text-gray-600 mt-1">Secure storage for all merchant agreements and contracts</p>
        </div>
        <button onclick="toggleAddContractForm()" class="bg-blue-600 text-white px-3 sm:px-4 py-2 rounded-md text-xs sm:text-sm font-semibold hover:bg-blue-700 transition">
            <i class="fas fa-plus-circle me-1"></i>
            <span id="addContractBtnText">Add Contract</span>
        </button>
    </div>

    <!-- Add Contract Form (Hidden by default) -->
    <div id="addContractForm" class="bg-white rounded-lg shadow-md p-3 sm:p-4 lg:p-6 mb-4" style="display: none;">
        <h2 class="text-lg font-semibold text-gray-900 mb-4">Add New Contract</h2>
        <form action="{{ route('admin.marketing.contracts.store') }}" method="POST" enctype="multipart/form-data">
            @csrf
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label class="text-sm font-medium text-gray-700">Merchant <span class="text-red-600">*</span></label>
                    <select name="merchant_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required>
                        <option value="">Select Merchant</option>
                        @foreach (\App\Models\Merchant::all() as $merchant)
                            <option value="{{ $merchant->id }}">{{ $merchant->name }}</option>
                        @endforeach
                    </select>
                </div>
                <div>
                    <label class="text-sm font-medium text-gray-700">Contract Type <span class="text-red-600">*</span></label>
                    <select name="contract_type" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required>
                        <option value="Service Agreement">Service Agreement</option>
                        <option value="Partnership">Partnership</option>
                        <option value="NDA">NDA</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
                <div>
                    <label class="text-sm font-medium text-gray-700">Start Date <span class="text-red-600">*</span></label>
                    <input type="date" name="start_date" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required>
                </div>
                <div>
                    <label class="text-sm font-medium text-gray-700">End Date <span class="text-red-600">*</span></label>
                    <input type="date" name="end_date" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required>
                </div>
                <div class="md:col-span-2">
                    <label class="text-sm font-medium text-gray-700">Upload Contract Document <span class="text-red-600">*</span></label>
                    <input type="file" name="document" accept=".pdf,.doc,.docx" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required>
                    <p class="text-xs text-gray-500 mt-1">Accepted formats: PDF, DOC, DOCX (Max 10MB)</p>
                </div>
            </div>
            <div class="mt-4">
                <label class="text-sm font-medium text-gray-700">Notes</label>
                <textarea name="notes" rows="3" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" placeholder="Add any additional notes..."></textarea>
            </div>
            <div class="flex gap-2 mt-4">
                <button type="submit" class="bg-blue-600 text-white px-4 py-2 rounded-md text-sm font-semibold hover:bg-blue-700 transition">
                    <i class="fas fa-save me-1"></i>
                    Save Contract
                </button>
                <button type="button" onclick="toggleAddContractForm()" class="bg-gray-200 text-gray-800 px-4 py-2 rounded-md text-sm font-semibold hover:bg-gray-300 transition">
                    Cancel
                </button>
            </div>
        </form>
    </div>

    <!-- Contracts Table -->
    <div class="bg-white rounded-lg shadow-md p-3 sm:p-4 lg:p-6">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Merchant</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Contract Type</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Start Date</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">End Date</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse ($contracts as $contract)
                        <tr class="hover:bg-gray-50">
                            <td class="px-4 py-3 whitespace-nowrap">
                                <div class="text-sm font-medium text-gray-900">{{ $contract->merchant->name }}</div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <div class="text-sm text-gray-900">{{ $contract->contract_type ?? 'Service Agreement' }}</div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <div class="text-sm text-gray-900">{{ \Carbon\Carbon::parse($contract->start_date)->format('M d, Y') }}</div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <div class="text-sm text-gray-900">{{ \Carbon\Carbon::parse($contract->end_date)->format('M d, Y') }}</div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                @php
                                    $isExpired = \Carbon\Carbon::parse($contract->end_date)->isPast();
                                    $isExpiringSoon = \Carbon\Carbon::parse($contract->end_date)->diffInDays(now()) < 30 && !$isExpired;
                                @endphp
                                @if($isExpired)
                                    <span class="px-2 py-1 text-xs font-semibold rounded-full bg-red-100 text-red-800">Expired</span>
                                @elseif($isExpiringSoon)
                                    <span class="px-2 py-1 text-xs font-semibold rounded-full bg-yellow-100 text-yellow-800">Expiring Soon</span>
                                @else
                                    <span class="px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">Active</span>
                                @endif
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm font-medium">
                                <div class="flex items-center gap-2">
                                    @if($contract->file_path)
                                        <a href="{{ Storage::url($contract->file_path) }}" target="_blank" class="text-green-600 hover:text-green-900">
                                            <i class="fas fa-file-pdf"></i> View
                                        </a>
                                    @endif
                                    <a href="{{ route('admin.marketing.contracts.edit', $contract->id) }}" class="text-blue-600 hover:text-blue-900">
                                        <i class="fas fa-edit"></i> Edit
                                    </a>
                                    <form action="{{ route('admin.marketing.contracts.destroy', $contract->id) }}" 
                                          method="POST" 
                                          class="inline-block"
                                          onsubmit="return confirm('Are you sure you want to delete this contract?');">
                                        @csrf
                                        @method('DELETE')
                                        <button type="submit" class="text-red-600 hover:text-red-900">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="px-4 py-8 text-center text-gray-500">
                                <i class="fas fa-file-contract fa-3x mb-3 text-gray-300"></i>
                                <p>No contracts found. Add your first contract to get started.</p>
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function toggleAddContractForm() {
    const form = document.getElementById('addContractForm');
    const btnText = document.getElementById('addContractBtnText');
    
    if (form.style.display === 'none') {
        form.style.display = 'block';
        btnText.textContent = 'Cancel';
        form.scrollIntoView({ behavior: 'smooth', block: 'start' });
    } else {
        form.style.display = 'none';
        btnText.textContent = 'Add Contract';
    }
}
</script>
@endsection
