@extends('layouts.admin')

@section('title', 'Bagging')

@section('content')
<div class="container mx-auto px-4 sm:px-6 lg:px-8 py-4">
    <div class="flex justify-between items-center mb-4">
        <h1 class="text-2xl font-bold text-gray-800 flex items-center">
            <i class="fa-regular fa-closed-captioning mr-2"></i> Bagging
        </h1>
        <div class="flex items-center space-x-4">
            <button class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-bell"></i>
            </button>
            <div class="w-8 h-8 bg-gray-300 rounded-full"></div>
        </div>
    </div>

    <div class="bg-white p-6 rounded-lg shadow-sm">
        <h2 class="text-lg font-semibold text-gray-900">Create Dispatch Bag</h2>
        <p class="mt-1 text-sm text-gray-600">Scan parcels to group them into a bag for dispatch. The system will auto-generate a Bag ID and Seal Number.</p>

        <form id="add-parcel-form" class="mt-4 flex items-center space-x-3">
            <div class="relative flex-grow">
                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                     <i class="fa-regular fa-barcode-read text-gray-400"></i>
                </div>
                <input type="text" id="parcel_tracking_id" name="parcel_tracking_id" class="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:placeholder-gray-400 focus:ring-1 focus:ring-blue-500 focus:border-blue-500 sm:text-sm" placeholder="Scan or enter parcel ID..." autocomplete="off">
            </div>
            <button type="submit" class="inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                <i class="fa-regular fa-magnifying-glass mr-2"></i> Add to Bag
            </button>
        </form>
    </div>

    <div id="bag-details-container" class="mt-6 bg-white p-6 rounded-lg shadow-sm hidden">
        <!-- Bag details will be injected here -->
    </div>

</div>
@endsection

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function () {
        const form = document.getElementById('add-parcel-form');
        const parcelInput = document.getElementById('parcel_tracking_id');
        const bagDetailsContainer = document.getElementById('bag-details-container');

        form.addEventListener('submit', function (e) {
            e.preventDefault();
            const trackingId = parcelInput.value;

            if (trackingId) {
                fetch('{{ route("admin.branch.bagging.add-parcel") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ 
                        parcel_tracking_id: trackingId 
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        parcelInput.value = ''; // Clear input
                        renderBagDetails(data.bag);
                    } else {
                        // Handle error
                        alert(data.message || 'An error occurred.');
                    }
                })
                .catch(error => console.error('Error:', error));
            }
        });

        function renderBagDetails(bag) {
            bagDetailsContainer.innerHTML = `
                <div class="flex justify-between items-center mb-4">
                    <div>
                        <h3 class="text-lg font-semibold text-gray-900">Bag ID: ${bag.bag_number}</h3>
                        <p class="text-sm text-gray-600">Destination: ${bag.destination_branch.name}</p> 
                    </div>
                    <button id="seal-bag-btn" data-bag-id="${bag.id}" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-green-600 hover:bg-green-700">Seal Bag</button>
                </div>
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tracking ID</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Recipient</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        ${bag.parcels.map(parcel => `
                            <tr>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">${parcel.tracking_id}</td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${parcel.recipient_name}</td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${parcel.status}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            bagDetailsContainer.classList.remove('hidden');

            const sealBagBtn = document.getElementById('seal-bag-btn');
            sealBagBtn.addEventListener('click', function () {
                const bagId = this.dataset.bagId;
                fetch(`/branch/bagging/${bagId}/seal`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        bagDetailsContainer.innerHTML = '<p class="text-green-600">Bag sealed successfully.</p>';
                        setTimeout(() => location.reload(), 2000);
                    } else {
                        alert('Failed to seal the bag.');
                    }
                })
                .catch(error => console.error('Error:', error));
            });
        }
    });
</script>
@endpush
