<?php

use App\Http\Controllers\AuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\ParcelController;
use App\Http\Controllers\ComplaintController;
use App\Http\Controllers\HomeController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
*/

// Public routes
Route::get('/', [HomeController::class, 'index'])->name('home');
Route::post('/track', [ParcelController::class, 'track'])->name('parcels.track');

// Authentication routes
Route::middleware('guest')->group(function () {
    Route::get('/login', [AuthController::class, 'showLogin'])->name('login');
    Route::post('/login', [AuthController::class, 'login']);
    Route::get('/register', [AuthController::class, 'showRegister'])->name('register');
    Route::post('/register', [AuthController::class, 'register']);
});

Route::post('/logout', [AuthController::class, 'logout'])->name('logout')->middleware('auth');

// Seller routes
Route::group(['middleware' => ['auth', 'role:Seller'], 'prefix' => 'seller', 'as' => 'seller.', 'namespace' => 'App\Http\Controllers\Seller'], function () {
    Route::get('/dashboard', 'DashboardController@sellerDashboard')->name('dashboard');

    // Manage Orders
    Route::resource('orders', 'OrderController');
    Route::get('transit-orders', 'OrderController@transit')->name('orders.transit');
    Route::get('under-verification', 'OrderController@verification')->name('orders.verification');
    Route::get('order-logs', 'OrderController@logs')->name('orders.logs');
    Route::get('airway-bills', 'OrderController@airwayBills')->name('orders.airway-bills');
    Route::post('print-airway-bills', 'OrderController@printAirwayBills')->name('orders.print-airway-bills');
    Route::get('bulk-upload', 'OrderController@bulkUpload')->name('orders.bulk-upload');
    Route::resource('complaints', 'ComplaintController');
    Route::post('complaints/search-parcel', 'ComplaintController@searchParcel')->name('complaints.search-parcel');

    // Load Sheet
    Route::get('load-sheet/generate', 'LoadSheetController@create')->name('load-sheet.create');
    Route::post('load-sheet', 'LoadSheetController@store')->name('load-sheet.store');
    Route::get('load-sheet/logs', 'LoadSheetController@index')->name('load-sheet.index');

    // Payments
    Route::get('payments/receipt', 'PaymentController@receipt')->name('payments.receipt');
    Route::get('payments/settlement-logs', 'PaymentController@settlements')->name('payments.settlements');
    Route::get('payments/tax-invoices', 'PaymentController@invoices')->name('payments.invoices');

    // Packaging
    Route::get('packaging/request', 'PackagingController@create')->name('packaging.create');
    Route::post('packaging', 'PackagingController@store')->name('packaging.store');
    Route::get('packaging/logs', 'PackagingController@index')->name('packaging.index');

    // Reports
    Route::get('reports/orders', 'ReportController@orders')->name('reports.orders');
    Route::get('reports/orders/pdf', 'ReportController@downloadOrdersPdf')->name('reports.orders.pdf');
    Route::get('reports/amounts', 'ReportController@amounts')->name('reports.amounts');
    Route::get('reports/financial', 'ReportController@financial')->name('reports.financial');
    Route::get('reports/status', 'ReportController@status')->name('reports.status');
    Route::get('reports/instock', 'ReportController@instock')->name('reports.instock');
    Route::get('reports/pickup', 'ReportController@pickup')->name('reports.pickup');
    Route::get('reports/destination', 'ReportController@destination')->name('reports.destination');
    Route::get('reports/complaints', 'ReportController@complaints')->name('reports.complaints');

    // Settings
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('profile', 'SettingsController@profile')->name('profile');
        Route::get('integrations', 'SettingsController@integrations')->name('integrations');
    });

    // Shipments & Parcels routes moved to non-namespaced seller group below to avoid FQCN conflicts

    // Others
    Route::get('integrations', 'OtherController@integrations')->name('others.integrations');
    Route::get('mobile-app', 'OtherController@mobileApp')->name('others.mobile-app');
    Route::get('company-news', 'OtherController@news')->name('others.news');
});

// Seller routes without namespace for FQCN controllers
Route::group(['middleware' => ['auth', 'role:Seller'], 'prefix' => 'seller', 'as' => 'seller.'], function () {
    // Shipments
    Route::get('shipments', [\App\Http\Controllers\ParcelController::class, 'index'])->name('shipments.index');
    Route::get('shipments/track', function () {
        return view('seller.shipments.track');
    })->name('shipments.track');

    // Parcels (limited resource for sellers)
    Route::resource('parcels', \App\Http\Controllers\ParcelController::class)->only(['index', 'create', 'store', 'show']);
});

// Admin routes
Route::middleware(['auth', 'role:Admin|Super Admin'])->prefix('admin')->name('admin.')->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'adminDashboard'])->name('dashboard');
    Route::post('/parcels/bulk-update', [ParcelController::class, 'bulkUpdate'])->name('parcels.bulk-update');
    Route::resource('parcels', ParcelController::class);
    Route::resource('complaints', ComplaintController::class);

    // Span module
    Route::resource('spans', App\Http\Controllers\Admin\SpanController::class);
    
    // Merchants Management
    Route::resource('merchants', App\Http\Controllers\Admin\MerchantController::class);
    Route::post('merchants/{merchant}/approve', [App\Http\Controllers\Admin\MerchantController::class, 'approve'])->name('merchants.approve');

    // Branch Management
    Route::resource('branches', App\Http\Controllers\Admin\BranchController::class);
    
    // Bank Management (Super Admin Only)
    Route::resource('banks', App\Http\Controllers\Admin\BankController::class);
    Route::get('banks/active/list', [App\Http\Controllers\Admin\BankController::class, 'getActiveBanks'])->name('banks.active');

    
    // Marketing Routes
    Route::prefix('marketing')->name('marketing.')->group(function () {
        // General Marketing pages
        Route::get('dashboard', [App\Http\Controllers\Admin\Marketing\MarketingController::class, 'dashboard'])->name('dashboard');
        Route::get('alerts', [App\Http\Controllers\Admin\Marketing\MarketingController::class, 'alerts'])->name('alerts');
        Route::get('alerts/data', [App\Http\Controllers\Admin\Marketing\MarketingController::class, 'alertsData'])->name('alerts.data');
        Route::post('alerts/{alert}/acknowledge', [App\Http\Controllers\Admin\Marketing\MarketingController::class, 'acknowledgeAlert'])->name('alerts.acknowledge');
        Route::post('alerts/{alert}/snooze', [App\Http\Controllers\Admin\Marketing\MarketingController::class, 'snoozeAlert'])->name('alerts.snooze');

        // Profile & Management Section
        Route::prefix('management')->name('management.')->group(function() {
            Route::get('profiles', [App\Http\Controllers\Admin\Marketing\MerchantProfileController::class, 'index'])->name('profiles');
            Route::get('hierarchy', [App\Http\Controllers\Admin\Marketing\ProfileController::class, 'hierarchy'])->name('hierarchy');
            Route::get('wallet-status', [App\Http\Controllers\Admin\Marketing\ProfileController::class, 'walletStatus'])->name('walletStatus');
            Route::get('shipment-summary', [App\Http\Controllers\Admin\Marketing\ProfileController::class, 'shipmentSummary'])->name('shipmentSummary');
        });

        // Rates Section
        Route::prefix('rates')->name('rates.')->group(function () {
            Route::get('pricing', [App\Http\Controllers\Admin\Marketing\RateController::class, 'pricing'])->name('pricing');
            Route::get('discounts', [App\Http\Controllers\Admin\Marketing\RateController::class, 'discounts'])->name('discounts');
            Route::get('zones', [App\Http\Controllers\Admin\Marketing\RateController::class, 'zones'])->name('zones');
        });
        
        // Documents Section
        Route::prefix('documents')->name('documents.')->group(function () {
            Route::get('/', [App\Http\Controllers\Admin\Marketing\DocumentController::class, 'index'])->name('index');
            Route::get('templates', [App\Http\Controllers\Admin\Marketing\DocumentController::class, 'templates'])->name('templates');
            Route::get('contracts', [App\Http\Controllers\Admin\Marketing\DocumentController::class, 'contracts'])->name('contracts');
            Route::get('compliance', [App\Http\Controllers\Admin\Marketing\DocumentController::class, 'compliance'])->name('compliance');
        });

        // Resource Controllers (must be defined after specific routes to avoid conflicts)
        Route::resource('leads', App\Http\Controllers\Admin\Marketing\LeadController::class);
        Route::resource('territories', App\Http\Controllers\Admin\Marketing\TerritoryController::class);
        Route::resource('contracts', App\Http\Controllers\Admin\Marketing\ContractController::class);
    });
    
    // Support Routes  
    Route::prefix('support')->name('support.')->group(function () {
        Route::get('/dashboard', [App\Http\Controllers\Admin\Support\SupportController::class, 'dashboard'])->name('dashboard');
        Route::get('/tickets', [App\Http\Controllers\Admin\Support\SupportController::class, 'tickets'])->name('tickets');

        // Tickets sub-routes
        Route::prefix('tickets')->name('tickets.')->group(function () {
            Route::get('/open', [App\Http\Controllers\Admin\Support\SupportController::class, 'open'])->name('open');
            Route::get('/pending', [App\Http\Controllers\Admin\Support\SupportController::class, 'pending'])->name('pending');
            Route::get('/resolved', [App\Http\Controllers\Admin\Support\SupportController::class, 'resolved'])->name('resolved');
            Route::get('/escalated', [App\Http\Controllers\Admin\Support\SupportController::class, 'escalated'])->name('escalated');
            Route::get('/categories', [App\Http\Controllers\Admin\Support\SupportController::class, 'categories'])->name('categories');
            Route::get('/sla', [App\Http\Controllers\Admin\Support\SupportController::class, 'sla'])->name('sla');
        });

        // Ticket CRUD (Admin\SupportController)
        Route::get('/tickets/create', [App\Http\Controllers\Admin\SupportController::class, 'create'])->name('tickets.create');
        Route::post('/tickets', [App\Http\Controllers\Admin\SupportController::class, 'store'])->name('tickets.store');
        Route::get('/tickets/{ticket}', [App\Http\Controllers\Admin\SupportController::class, 'show'])->name('tickets.show');
        Route::get('/tickets/{ticket}/edit', [App\Http\Controllers\Admin\SupportController::class, 'edit'])->name('tickets.edit');
        Route::put('/tickets/{ticket}', [App\Http\Controllers\Admin\SupportController::class, 'update'])->name('tickets.update');
        Route::delete('/tickets/{ticket}', [App\Http\Controllers\Admin\SupportController::class, 'destroy'])->name('tickets.destroy');
        Route::post('/tickets/{ticket}/assign', [App\Http\Controllers\Admin\SupportController::class, 'assign'])->name('tickets.assign');
        Route::post('/tickets/{ticket}/reply', [App\Http\Controllers\Admin\SupportController::class, 'addReply'])->name('tickets.reply');
    });
    
    // Riders Routes
    Route::prefix('riders')->name('riders.')->group(function () {
        Route::get('/dashboard', [App\Http\Controllers\Admin\Rider\RiderController::class, 'dashboard'])->name('dashboard');
        Route::get('/operations', [App\Http\Controllers\Admin\Rider\RiderController::class, 'operations'])->name('operations');
    });
    
    // HR Routes
    Route::prefix('hr')->name('hr.')->group(function () {
        Route::get('/dashboard', [App\Http\Controllers\Admin\HR\HRController::class, 'dashboard'])->name('dashboard');
        Route::get('/employees', [App\Http\Controllers\Admin\HR\HRController::class, 'employees'])->name('employees');
    });
    
    
    // System Settings
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/users', [App\Http\Controllers\Admin\Setting\SettingController::class, 'users'])->name('users');
        Route::get('/security', [App\Http\Controllers\Admin\Setting\SettingController::class, 'security'])->name('security');
        Route::get('/users/create', [App\Http\Controllers\Admin\Setting\SettingController::class, 'create'])->name('users.create');
        Route::post('/users', [App\Http\Controllers\Admin\Setting\SettingController::class, 'store'])->name('users.store');
        Route::get('/users/{user}/edit', [App\Http\Controllers\Admin\Setting\SettingController::class, 'edit'])->name('users.edit');
        Route::put('/users/{user}', [App\Http\Controllers\Admin\Setting\SettingController::class, 'update'])->name('users.update');
    });
});

// Branch routes
Route::group(['middleware' => ['auth', 'role:Branch Operator|Admin|Super Admin'], 'prefix' => 'branch', 'as' => 'admin.branch.'], function () {
    Route::get('/dashboard', [App\Http\Controllers\Admin\Branch\BranchController::class, 'dashboard'])->name('dashboard');
    Route::get('/bagging', [App\Http\Controllers\Branch\BaggingController::class, 'index'])->name('bagging');
    Route::post('/bagging', [App\Http\Controllers\Branch\BaggingController::class, 'store'])->name('bagging.store');
    Route::post('/bagging/add-parcel', [App\Http\Controllers\Branch\BaggingController::class, 'addParcel'])->name('bagging.add-parcel');
    Route::put('/bagging/{bag}/seal', [App\Http\Controllers\Branch\BaggingController::class, 'seal'])->name('bagging.seal');
    Route::get('/unbagging', [App\Http\Controllers\Branch\UnbaggingController::class, 'index'])->name('unbagging');
    Route::post('/unbagging/find-bag', [App\Http\Controllers\Branch\UnbaggingController::class, 'findBag'])->name('unbagging.find-bag');
    Route::put('/unbagging/{bag}/receive', [App\Http\Controllers\Branch\UnbaggingController::class, 'receiveBag'])->name('unbagging.receive');
    Route::get('/pickup-orders', [App\Http\Controllers\Branch\PickupOrdersController::class, 'index'])->name('pickup-orders');
    Route::put('/pickup-orders/{parcel}/mark-as-picked-up', [App\Http\Controllers\Branch\PickupOrdersController::class, 'markAsPickedUp'])->name('pickup-orders.mark-as-picked-up');
    Route::get('/receive-shipments', [App\Http\Controllers\Branch\ReceiveShipmentsController::class, 'index'])->name('receive-shipments');
    Route::post('/receive-shipments/scan', [App\Http\Controllers\Branch\ReceiveShipmentsController::class, 'scan'])->name('receive-shipments.scan');
    Route::get('/dispatch-shipments', [App\Http\Controllers\Branch\DispatchShipmentsController::class, 'index'])->name('dispatch-shipments');
    Route::put('/dispatch-shipments/{parcel}/assign-rider', [App\Http\Controllers\Branch\DispatchShipmentsController::class, 'assignRider'])->name('dispatch-shipments.assign-rider');
    Route::get('/successful-deliveries', [App\Http\Controllers\Branch\SuccessfulDeliveriesController::class, 'index'])->name('successful-deliveries');
    Route::get('/deliveries-assigned', [App\Http\Controllers\Branch\DeliveriesAssignedController::class, 'index'])->name('deliveries-assigned');
    Route::get('/failed-deliveries', [App\Http\Controllers\Branch\FailedDeliveriesController::class, 'index'])->name('failed-deliveries');
    Route::get('/reattempt-shipment', [App\Http\Controllers\Branch\ReattemptShipmentController::class, 'index'])->name('reattempt-shipment');
    Route::get('/return-shipments', [App\Http\Controllers\Branch\ReturnShipmentsController::class, 'index'])->name('return-shipments');
    Route::get('/forward-to-origin', [App\Http\Controllers\Branch\ForwardToOriginController::class, 'index'])->name('forward-to-origin');
    Route::get('/shipment-tracking', [App\Http\Controllers\Branch\ShipmentTrackingController::class, 'index'])->name('shipment-tracking');
    Route::get('/search-bag', [App\Http\Controllers\Branch\SearchBagController::class, 'index'])->name('search-bag');
    Route::get('/lost-hold', [App\Http\Controllers\Branch\LostHoldController::class, 'index'])->name('lost-hold');
    Route::get('/riders/onboarding', [App\Http\Controllers\Admin\Branch\RiderController::class, 'onboarding'])->name('riders.onboarding');
    Route::get('/riders/daily-load', [App\Http\Controllers\Admin\Branch\RiderController::class, 'dailyLoad'])->name('riders.daily-load');
    Route::get('/riders/cod-assigned', [App\Http\Controllers\Admin\Branch\RiderController::class, 'codAssigned'])->name('riders.cod-assigned');
    Route::get('/finance/rider-handover', [App\Http\Controllers\Admin\Branch\FinanceController::class, 'riderHandover'])->name('finance.rider-handover');
    Route::get('/finance/pending-cod', [App\Http\Controllers\Admin\Branch\FinanceController::class, 'pendingCod'])->name('finance.pending-cod');
    Route::get('/finance/variance-report', [App\Http\Controllers\Admin\Branch\FinanceController::class, 'varianceReport'])->name('finance.variance-report');
    Route::get('/cashier/collect-cod', [App\Http\Controllers\Admin\Branch\CashierController::class, 'collectCod'])->name('cashier.collect-cod');
});


// Finance routes (separated from admin to avoid route conflicts)
Route::middleware(['auth', 'role:Finance|Super Admin'])->prefix('finance')->name('finance.')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Admin\FinanceController::class, 'dashboard'])->name('dashboard');
    Route::get('/expenses', [App\Http\Controllers\Admin\FinanceController::class, 'expenses'])->name('expenses');
    Route::get('/payroll', [App\Http\Controllers\Admin\FinanceController::class, 'payroll'])->name('payroll');
});
