'use server';

/**
 * @fileOverview An AI agent to predict the estimated delivery time (ETA) for parcels.
 *
 * - predictDeliveryETA - A function that predicts the delivery ETA for a given parcel.
 * - PredictDeliveryETAInput - The input type for the predictDeliveryETA function.
 * - PredictDeliveryETAOutput - The return type for the predictDeliveryETA function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const PredictDeliveryETAInputSchema = z.object({
  parcelId: z.string().describe('The unique identifier of the parcel.'),
  city: z.string().describe('The destination city of the parcel.'),
  zone: z.string().describe('The destination zone of the parcel within the city.'),
  bookingDate: z.string().describe('The date when the parcel was booked (ISO format).'),
  historicalData: z.string().describe('JSON array of historical delivery data for similar parcels, each object should have city, zone, deliveryTime properties.'),
});
export type PredictDeliveryETAInput = z.infer<typeof PredictDeliveryETAInputSchema>;

const PredictDeliveryETAOutputSchema = z.object({
  estimatedDeliveryTime: z.string().describe('The estimated delivery time for the parcel (ISO format).'),
  confidence: z.number().describe('A confidence score (0-1) indicating the reliability of the ETA prediction.'),
  reasoning: z.string().describe('The reasoning behind the ETA prediction, including factors considered.'),
});
export type PredictDeliveryETAOutput = z.infer<typeof PredictDeliveryETAOutputSchema>;

export async function predictDeliveryETA(input: PredictDeliveryETAInput): Promise<PredictDeliveryETAOutput> {
  return predictDeliveryETAFlow(input);
}

const prompt = ai.definePrompt({
  name: 'predictDeliveryETAPrompt',
  input: {schema: PredictDeliveryETAInputSchema},
  output: {schema: PredictDeliveryETAOutputSchema},
  prompt: `You are an expert logistics analyst specializing in delivery time prediction.

  Given the following information about a parcel, and historical data of similar parcels, predict the estimated delivery time (ETA).
  Consider factors such as destination city, zone, booking date, and historical delivery times to provide an accurate prediction.

  Parcel Information:
  - Parcel ID: {{{parcelId}}}
  - City: {{{city}}}
  - Zone: {{{zone}}}
  - Booking Date: {{{bookingDate}}}

  Historical Data:
  {{{historicalData}}}

  Provide the estimated delivery time in ISO format, a confidence score (0-1), and a brief explanation of your reasoning.
  Ensure the response is correctly formatted JSON object matching the schema. Focus on providing accurate delivery time estimates based on available data and avoid hallucination.
  `,
});

const predictDeliveryETAFlow = ai.defineFlow(
  {
    name: 'predictDeliveryETAFlow',
    inputSchema: PredictDeliveryETAInputSchema,
    outputSchema: PredictDeliveryETAOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
