
'use client';

import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { mockParcels } from '@/lib/data';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { formatDistanceToNow } from 'date-fns';

type ReattemptParcel = Parcel & {
    failureReason: string;
    attemptCount: number;
    lastAttempt: string;
};

export default function AdminBranchAttemptsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const reattemptParcels: ReattemptParcel[] = mockParcels
    .filter(p => p.status === 'Failed' || p.status === 'Attempted')
    .slice(0, 4)
    .map((p, i) => ({
        ...p,
        failureReason: i % 2 === 0 ? 'Customer not available' : 'Address incorrect',
        attemptCount: i % 2 + 1,
        lastAttempt: new Date(Date.now() - (i + 1) * 12 * 60 * 60 * 1000).toISOString(),
    }));

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="branch-operator">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Reattempt Shipments</CardTitle>
              <CardDescription>
                Review shipments that failed delivery and schedule a new attempt.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Parcel ID</TableHead>
                            <TableHead>Customer</TableHead>
                            <TableHead>Failure Reason</TableHead>
                            <TableHead>Last Attempt</TableHead>
                            <TableHead>Attempt #</TableHead>
                            <TableHead className="text-right">Action</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {reattemptParcels.map(parcel => (
                            <TableRow key={parcel.id}>
                                <TableCell className="font-medium">{parcel.id}</TableCell>
                                <TableCell>{parcel.recipientName}</TableCell>
                                <TableCell>{parcel.failureReason}</TableCell>
                                <TableCell>{formatDistanceToNow(new Date(parcel.lastAttempt), { addSuffix: true })}</TableCell>
                                <TableCell>
                                    <Badge variant={parcel.attemptCount > 1 ? "destructive" : "secondary"}>
                                        {parcel.attemptCount}
                                    </Badge>
                                </TableCell>
                                <TableCell className="text-right">
                                    <Button size="sm" variant="outline">
                                        Schedule Reattempt
                                    </Button>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
