
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Rider } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Tabs, TabsList, TabsTrigger, TabsContent } from '@/components/ui/tabs';
import { UserCheck, Upload, FileText, ListChecks, User, ArrowRight, DollarSign, Save, PlusCircle, Trash2 } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { mockRiders } from '@/lib/data';


const sopItems = [
    { id: 'sop_uniform', label: 'Uniform Policy & Grooming Standards' },
    { id: 'sop_cod', label: 'Cash on Delivery (COD) Handling Procedure' },
    { id: 'sop_customer', label: 'Customer Interaction & Service Protocol' },
    { id: 'sop_app', label: 'Mobile App Usage & Delivery Workflow' },
    { id: 'sop_safety', label: 'Road Safety & Vehicle Maintenance' },
    { id: 'sop_returns', label: 'Return to Origin (RTO) Process' },
];

type Allowance = {
  id: string;
  name: string;
  value: number;
};

type ExtraField = {
  id: string;
  label: string;
  value: string;
};

type ExtraDocumentField = {
    id: string;
    label: string;
};

type ExtraSopField = {
    id: string;
    label: string;
    checked: boolean;
};


export default function AdminBranchRiderManagementPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [currentStep, setCurrentStep] = useState('personal');
  const [newRider, setNewRider] = useState<Partial<Rider>>({});
  const [checkedSops, setCheckedSops] = useState<Record<string, boolean>>({});
  const { toast } = useToast();
  const [additionalAllowances, setAdditionalAllowances] = useState<Allowance[]>([]);
  const [extraFields, setExtraFields] = useState<ExtraField[]>([]);
  const [extraDocumentFields, setExtraDocumentFields] = useState<ExtraDocumentField[]>([]);
  const [extraSopFields, setExtraSopFields] = useState<ExtraSopField[]>([]);

  const handleNextStep = (next: string) => {
    setCurrentStep(next);
  }

  const handleFinishOnboarding = () => {
    toast({
        title: 'Onboarding Complete!',
        description: `${newRider.name} has been successfully onboarded.`,
    });
    // Reset state
    setCurrentStep('personal');
    setNewRider({});
    setCheckedSops({});
    setAdditionalAllowances([]);
    setExtraFields([]);
    setExtraDocumentFields([]);
    setExtraSopFields([]);
  }

  const handleAddAllowance = () => {
    setAdditionalAllowances(prev => [...prev, { id: `allowance-${Date.now()}`, name: '', value: 0 }]);
  };

  const handleRemoveAllowance = (id: string) => {
    setAdditionalAllowances(prev => prev.filter(a => a.id !== id));
  };

  const handleAllowanceChange = (id: string, field: 'name' | 'value', value: string | number) => {
    setAdditionalAllowances(prev => prev.map(a => a.id === id ? { ...a, [field]: value } : a));
  };
  
  const handleAddExtraField = () => {
    setExtraFields(prev => [...prev, { id: `extra-${Date.now()}`, label: '', value: '' }]);
  };

  const handleRemoveExtraField = (id: string) => {
    setExtraFields(prev => prev.filter(f => f.id !== id));
  };

  const handleExtraFieldChange = (id: string, field: 'label' | 'value', value: string) => {
    setExtraFields(prev => prev.map(f => f.id === id ? { ...f, [field]: value } : a));
  };

  const handleAddExtraDocumentField = () => {
    setExtraDocumentFields(prev => [...prev, { id: `doc-${Date.now()}`, label: '' }]);
  };

  const handleRemoveExtraDocumentField = (id: string) => {
    setExtraDocumentFields(prev => prev.filter(d => d.id !== id));
  };

  const handleExtraDocumentFieldChange = (id: string, value: string) => {
    setExtraDocumentFields(prev => prev.map(d => d.id === id ? { ...d, label: value } : d));
  };
  
  const handleAddExtraSopField = () => {
    setExtraSopFields(prev => [...prev, { id: `sop-${Date.now()}`, label: '', checked: false }]);
  };

  const handleRemoveExtraSopField = (id: string) => {
    setExtraSopFields(prev => prev.filter(s => s.id !== id));
  };

  const handleExtraSopFieldChange = (id: string, value: string) => {
    setExtraSopFields(prev => prev.map(s => s.id === id ? { ...s, label: value } : s));
  };
  
  const handleExtraSopFieldCheck = (id: string, checked: boolean) => {
    setExtraSopFields(prev => prev.map(s => s.id === id ? { ...s, checked: checked } : s));
  };


  const isSopComplete = sopItems.every(item => checkedSops[item.id]) && extraSopFields.every(item => item.checked);


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="branch-operator">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Rider Onboarding</CardTitle>
              <CardDescription>
                Follow the steps to onboard a new rider into the system.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <Tabs value={currentStep} onValueChange={setCurrentStep} className="w-full">
                    <TabsList className="grid w-full grid-cols-5">
                        <TabsTrigger value="personal" className="gap-2">
                           <User className="h-4 w-4"/> Personal Details
                        </TabsTrigger>
                        <TabsTrigger value="documents" className="gap-2">
                           <FileText className="h-4 w-4"/> Documents
                        </TabsTrigger>
                        <TabsTrigger value="salary" className="gap-2">
                           <DollarSign className="h-4 w-4"/> Salary Setup
                        </TabsTrigger>
                        <TabsTrigger value="sop" className="gap-2">
                           <ListChecks className="h-4 w-4"/> SOP Checklist
                        </TabsTrigger>
                         <TabsTrigger value="review" className="gap-2">
                           <UserCheck className="h-4 w-4"/> Review & Submit
                        </TabsTrigger>
                    </TabsList>
                    
                    <TabsContent value="personal" className="mt-6">
                        <Card>
                            <CardHeader>
                                <CardTitle>Step 1: Personal Details</CardTitle>
                                <CardDescription>Enter the basic information for the new rider.</CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="flex items-center gap-4">
                                    <Avatar className="h-20 w-20">
                                        <AvatarImage src={newRider.avatar} />
                                        <AvatarFallback><User className="h-10 w-10 text-muted-foreground"/></AvatarFallback>
                                    </Avatar>
                                    <Input type="file" className="max-w-xs"/>
                                </div>
                                <div className="grid md:grid-cols-2 gap-4">
                                    <div>
                                        <Label htmlFor="name">Rider Name</Label>
                                        <Input id="name" placeholder="e.g., Ali Khan" onChange={e => setNewRider(p => ({...p, name: e.target.value}))} />
                                    </div>
                                    <div>
                                        <Label htmlFor="phone">Phone Number</Label>
                                        <Input id="phone" placeholder="e.g., 03001234567" onChange={e => setNewRider(p => ({...p, phone: e.target.value}))}/>
                                    </div>
                                    <div>
                                        <Label htmlFor="zone">Assign Zone</Label>
                                        <Select onValueChange={val => setNewRider(p => ({...p, zone: val}))}>
                                            <SelectTrigger><SelectValue placeholder="Select a zone" /></SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="A">Zone A</SelectItem>
                                                <SelectItem value="B">Zone B</SelectItem>
                                                <SelectItem value="C">Zone C</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                </div>
                                 {extraFields.map((field, index) => (
                                    <div key={field.id} className="grid grid-cols-[1fr_1fr_auto] gap-2 items-end">
                                        <div>
                                            {index === 0 && <Label>Custom Field</Label>}
                                            <Input 
                                                placeholder="Field Name" 
                                                value={field.label} 
                                                onChange={e => handleExtraFieldChange(field.id, 'label', e.target.value)}
                                            />
                                        </div>
                                        <div>
                                             {index === 0 && <Label>Value</Label>}
                                            <Input 
                                                placeholder="Field Value" 
                                                value={field.value}
                                                onChange={e => handleExtraFieldChange(field.id, 'value', e.target.value)}
                                            />
                                        </div>
                                        <Button variant="ghost" size="icon" onClick={() => handleRemoveExtraField(field.id)}>
                                            <Trash2 className="h-4 w-4 text-red-500"/>
                                        </Button>
                                    </div>
                                ))}

                                <Button type="button" variant="outline" onClick={handleAddExtraField}>
                                    <PlusCircle className="mr-2 h-4 w-4" /> Add Field
                                </Button>
                            </CardContent>
                             <CardFooter className="justify-end">
                                <Button onClick={() => handleNextStep('documents')}>Next <ArrowRight className="ml-2 h-4 w-4"/></Button>
                            </CardFooter>
                        </Card>
                    </TabsContent>
                    
                    <TabsContent value="documents" className="mt-6">
                         <Card>
                            <CardHeader>
                                <CardTitle>Step 2: Document Upload</CardTitle>
                                <CardDescription>Upload necessary documents for verification.</CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-6">
                                <div className="space-y-2">
                                    <Label htmlFor="cnic-front">CNIC (Front Side)</Label>
                                    <div className="flex items-center gap-2 max-w-md">
                                        <Input id="cnic-front" type="file" />
                                        <Button variant="secondary"><Upload className="mr-2 h-4 w-4"/>Upload</Button>
                                    </div>
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="cnic-back">CNIC (Back Side)</Label>
                                    <div className="flex items-center gap-2 max-w-md">
                                        <Input id="cnic-back" type="file" />
                                        <Button variant="secondary"><Upload className="mr-2 h-4 w-4"/>Upload</Button>
                                    </div>
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="license">Driver's License</Label>
                                     <div className="flex items-center gap-2 max-w-md">
                                        <Input id="license" type="file" />
                                        <Button variant="secondary"><Upload className="mr-2 h-4 w-4"/>Upload</Button>
                                    </div>
                                </div>

                                {extraDocumentFields.map((field) => (
                                    <div key={field.id} className="space-y-2">
                                        <div className="flex items-center gap-2">
                                            <Input 
                                                placeholder="Document Name" 
                                                value={field.label}
                                                onChange={(e) => handleExtraDocumentFieldChange(field.id, e.target.value)}
                                            />
                                            <Button variant="ghost" size="icon" onClick={() => handleRemoveExtraDocumentField(field.id)}>
                                                <Trash2 className="h-4 w-4 text-red-500" />
                                            </Button>
                                        </div>
                                        <div className="flex items-center gap-2 max-w-md">
                                            <Input id={`doc-${field.id}`} type="file" />
                                            <Button variant="secondary"><Upload className="mr-2 h-4 w-4"/>Upload</Button>
                                        </div>
                                    </div>
                                ))}

                                <Button type="button" variant="outline" onClick={handleAddExtraDocumentField}>
                                    <PlusCircle className="mr-2 h-4 w-4" /> Add Field
                                </Button>

                            </CardContent>
                             <CardFooter className="justify-between">
                                <Button variant="outline" onClick={() => handleNextStep('personal')}>Back</Button>
                                <Button onClick={() => handleNextStep('salary')}>Next <ArrowRight className="ml-2 h-4 w-4"/></Button>
                            </CardFooter>
                        </Card>
                    </TabsContent>

                    <TabsContent value="salary" className="mt-6">
                         <Card>
                            <CardHeader>
                                <CardTitle>Step 3: Salary & Commission</CardTitle>
                                <CardDescription>Configure salary structures, allowances, and commission rates.</CardDescription>
                            </CardHeader>
                             <CardContent className="space-y-6">
                                <div className="max-w-2xl mx-auto space-y-4">
                                    <div>
                                        <Label htmlFor="basic-pay">Basic Pay (PKR)</Label>
                                        <Input id="basic-pay" type="number" placeholder="e.g., 25000" />
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <Label htmlFor="fuel-allowance">Fuel Allowance (PKR)</Label>
                                            <Input id="fuel-allowance" type="number" placeholder="e.g., 5000" />
                                        </div>
                                        <div>
                                            <Label htmlFor="mobile-allowance">Mobile Allowance (PKR)</Label>
                                            <Input id="mobile-allowance" type="number" placeholder="e.g., 1000" />
                                        </div>
                                    </div>
                                    
                                    {additionalAllowances.map((allowance, index) => (
                                        <div key={allowance.id} className="grid grid-cols-[1fr_1fr_auto] gap-2 items-end">
                                            <div>
                                                {index === 0 && <Label>Additional Allowance</Label>}
                                                <Input 
                                                    placeholder="Allowance Name" 
                                                    value={allowance.name} 
                                                    onChange={e => handleAllowanceChange(allowance.id, 'name', e.target.value)}
                                                />
                                            </div>
                                            <div>
                                                {index === 0 && <Label>Amount (PKR)</Label>}
                                                <Input 
                                                    type="number" 
                                                    placeholder="e.g., 500" 
                                                    value={allowance.value}
                                                    onChange={e => handleAllowanceChange(allowance.id, 'value', e.target.valueAsNumber)}
                                                />
                                            </div>
                                            <Button variant="ghost" size="icon" onClick={() => handleRemoveAllowance(allowance.id)}>
                                                <Trash2 className="h-4 w-4 text-red-500"/>
                                            </Button>
                                        </div>
                                    ))}

                                    <Button type="button" variant="outline" onClick={handleAddAllowance}>
                                        <PlusCircle className="mr-2 h-4 w-4" /> Add Allowance
                                    </Button>

                                    <div>
                                        <Label htmlFor="commission-rate">Commission Rate (%) per Delivery</Label>
                                        <Input id="commission-rate" type="number" placeholder="e.g., 5" />
                                    </div>
                                </div>
                            </CardContent>
                            <CardFooter className="justify-between">
                                <Button variant="outline" onClick={() => handleNextStep('documents')}>Back</Button>
                                <Button onClick={() => handleNextStep('sop')}>
                                    Next <ArrowRight className="ml-2 h-4 w-4"/>
                                </Button>
                            </CardFooter>
                        </Card>
                    </TabsContent>

                    <TabsContent value="sop" className="mt-6">
                         <Card>
                            <CardHeader>
                                <CardTitle>Step 4: SOP Checklist</CardTitle>
                                <CardDescription>Confirm that the rider has been trained on all Standard Operating Procedures.</CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                {sopItems.map(item => (
                                    <div key={item.id} className="flex items-center space-x-2 p-3 rounded-md border bg-muted/50">
                                        <Checkbox 
                                            id={item.id} 
                                            checked={checkedSops[item.id] || false}
                                            onCheckedChange={(checked) => setCheckedSops(p => ({...p, [item.id]: !!checked}))}
                                        />
                                        <Label htmlFor={item.id} className="flex-1 cursor-pointer">{item.label}</Label>
                                    </div>
                                ))}

                                {extraSopFields.map((field) => (
                                    <div key={field.id} className="flex items-center space-x-2 p-3 rounded-md border bg-muted/50">
                                        <Checkbox
                                            id={field.id}
                                            checked={field.checked}
                                            onCheckedChange={(c) => handleExtraSopFieldCheck(field.id, c as boolean)}
                                        />
                                        <Input
                                            placeholder="Custom SOP Item"
                                            value={field.label}
                                            onChange={(e) => handleExtraSopFieldChange(field.id, e.target.value)}
                                            className="flex-1"
                                        />
                                        <Button variant="ghost" size="icon" onClick={() => handleRemoveExtraSopField(field.id)}>
                                            <Trash2 className="h-4 w-4 text-red-500" />
                                        </Button>
                                    </div>
                                ))}
                                <Button type="button" variant="outline" onClick={handleAddExtraSopField}>
                                    <PlusCircle className="mr-2 h-4 w-4" /> Add SOP Item
                                </Button>

                            </CardContent>
                            <CardFooter className="justify-between">
                                <Button variant="outline" onClick={() => handleNextStep('salary')}>Back</Button>
                                <Button onClick={() => handleNextStep('review')} disabled={!isSopComplete}>
                                    Next <ArrowRight className="ml-2 h-4 w-4"/>
                                </Button>
                            </CardFooter>
                        </Card>
                    </TabsContent>
                    
                    <TabsContent value="review" className="mt-6">
                         <Card>
                            <CardHeader>
                                <CardTitle>Step 5: Review & Submit</CardTitle>
                                <CardDescription>Review all the information before completing the onboarding process.</CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-4">
                               <p><strong>Name:</strong> {newRider.name || 'N/A'}</p>
                               <p><strong>Phone:</strong> {newRider.phone || 'N/A'}</p>
                               <p><strong>Zone:</strong> {newRider.zone || 'N/A'}</p>
                               <p><strong>Documents:</strong> 3 uploaded</p>
                               <p><strong>SOPs Confirmed:</strong> {Object.values(checkedSops).filter(Boolean).length} of {sopItems.length}</p>
                            </CardContent>
                            <CardFooter className="justify-between">
                                <Button variant="outline" onClick={() => handleNextStep('sop')}>Back</Button>
                                <Button onClick={handleFinishOnboarding}>
                                    <UserCheck className="mr-2 h-4 w-4"/> Finish Onboarding
                                </Button>
                            </CardFooter>
                        </Card>
                    </TabsContent>

                </Tabs>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
