
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Search, Package, Fingerprint } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import { mockParcels } from '@/lib/data';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';

type Bag = {
    id: string;
    sealId: string;
    parcels: Parcel[];
    origin: string;
    dispatchDate: string;
};

const mockBags: Bag[] = [
    { id: 'BAG-123456', sealId: 'SEAL-1234', parcels: mockParcels.slice(0,3), origin: 'Lahore Hub', dispatchDate: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString() },
    { id: 'BAG-789012', sealId: 'SEAL-5678', parcels: mockParcels.slice(3,5), origin: 'Karachi Central', dispatchDate: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString() },
];


export default function AdminBranchShipmentBagIdPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [bagIdInput, setBagIdInput] = useState('');
  const [searchedBag, setSearchedBag] = useState<Bag | null>(null);
  const { toast } = useToast();

  const handleSearch = () => {
    if (!bagIdInput) return;
    const bag = mockBags.find(b => b.id.toLowerCase() === bagIdInput.toLowerCase());
    
    if (bag) {
        setSearchedBag(bag);
    } else {
        setSearchedBag(null);
        toast({
            variant: 'destructive',
            title: 'Not Found',
            description: 'No shipment bag found with this ID.',
        });
    }
  };


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="branch-operator">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Search Shipment Bag</CardTitle>
              <CardDescription>
                Enter a Bag ID to view its details and contents.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <div className="flex items-center gap-2 max-w-lg mx-auto">
                    <div className="relative flex-1">
                        <Fingerprint className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                        <Input 
                            placeholder="Scan or enter Bag ID..." 
                            className="pl-10 h-12 text-lg" 
                            value={bagIdInput}
                            onChange={(e) => setBagIdInput(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                        />
                    </div>
                    <Button size="lg" className="h-12" onClick={handleSearch}>
                        <Search className="mr-2 h-5 w-5" /> Find Bag
                    </Button>
                </div>
            </CardContent>
          </Card>
          
          {searchedBag && (
            <Card>
                <CardHeader>
                    <div className="flex justify-between items-start">
                        <div>
                            <CardTitle>Bag Details: {searchedBag.id}</CardTitle>
                            <CardDescription>
                                From: {searchedBag.origin} | Dispatched: {new Date(searchedBag.dispatchDate).toLocaleDateString()}
                            </CardDescription>
                        </div>
                        <div className="text-right">
                           <Badge variant="secondary" className="text-base">Seal: {searchedBag.sealId}</Badge>
                           <p className="text-sm text-muted-foreground mt-1">
                                <Package className="inline h-4 w-4 mr-1"/>
                                {searchedBag.parcels.length} Parcels
                           </p>
                        </div>
                    </div>
                </CardHeader>
                <CardContent>
                     <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Parcel ID</TableHead>
                                <TableHead>Recipient</TableHead>
                                <TableHead>Destination</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead className="text-right">COD Amount</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {searchedBag.parcels.map(p => (
                                <TableRow key={p.id}>
                                    <TableCell>{p.id}</TableCell>
                                    <TableCell>{p.recipientName}</TableCell>
                                    <TableCell>{p.city}</TableCell>
                                    <TableCell>
                                        <Badge variant="outline">{p.status}</Badge>
                                    </TableCell>
                                    <TableCell className="text-right">PKR {p.codAmount.toFixed(2)}</TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
          )}

        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
