
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, User, UserRole } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Button } from '@/components/ui/button';
import { DatePicker } from '@/components/ui/date-picker';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { useToast } from '@/hooks/use-toast';
import { Save } from 'lucide-react';
import { format } from 'date-fns';

const mockStaff: User[] = [
    { id: 'staff-1', name: 'Ahmed Hassan', email: 'ahmed.h@branch.com', role: 'branch-operator', companyName: 'Multan Branch', avatar: 'https://picsum.photos/seed/staff1/100/100' },
    { id: 'staff-2', name: 'Aisha Bibi', email: 'aisha.b@branch.com', role: 'branch-operator', companyName: 'Multan Branch', avatar: 'https://picsum.photos/seed/staff2/100/100' },
    { id: 'staff-3', name: 'Bilal Khan', email: 'bilal.k@branch.com', role: 'branch-operator', companyName: 'Multan Branch', avatar: 'https://picsum.photos/seed/staff3/100/100' },
];

type AttendanceStatus = 'Present' | 'Absent' | 'Leave';
type AttendanceRecord = Record<string, AttendanceStatus>;


export default function AdminBranchStaffAttendancePage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [selectedDate, setSelectedDate] = useState<Date | undefined>(new Date());
  const [attendance, setAttendance] = useState<AttendanceRecord>({});
  const { toast } = useToast();

  const handleStatusChange = (staffId: string, status: AttendanceStatus) => {
    setAttendance(prev => ({ ...prev, [staffId]: status }));
  };
  
  const handleSave = () => {
    if (Object.keys(attendance).length !== mockStaff.length) {
        toast({
            variant: 'destructive',
            title: 'Incomplete Attendance',
            description: 'Please mark the attendance for all staff members before saving.',
        });
        return;
    }
    toast({
        title: 'Attendance Saved',
        description: `Attendance for ${format(selectedDate!, 'dd MMMM, yyyy')} has been successfully recorded.`,
    });
  }

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="branch-operator">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
                <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                    <div>
                        <CardTitle>Staff Attendance</CardTitle>
                        <CardDescription>
                            Mark and track daily attendance for your branch staff.
                        </CardDescription>
                    </div>
                    <div className="flex items-center gap-2">
                        <Label>Select Date:</Label>
                        <DatePicker date={selectedDate} setDate={setSelectedDate} />
                    </div>
                </div>
            </CardHeader>
            <CardContent>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Staff Member</TableHead>
                            <TableHead>Role</TableHead>
                            <TableHead className="text-center">Status</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {mockStaff.map(staff => (
                            <TableRow key={staff.id}>
                                <TableCell>
                                    <div className="flex items-center gap-3">
                                        <Avatar>
                                            <AvatarImage src={staff.avatar} />
                                            <AvatarFallback>{staff.name.charAt(0)}</AvatarFallback>
                                        </Avatar>
                                        <div>
                                            <p className="font-medium">{staff.name}</p>
                                            <p className="text-sm text-muted-foreground">{staff.email}</p>
                                        </div>
                                    </div>
                                </TableCell>
                                <TableCell>Branch Operator</TableCell>
                                <TableCell>
                                    <RadioGroup
                                        value={attendance[staff.id] || ''}
                                        onValueChange={(value) => handleStatusChange(staff.id, value as AttendanceStatus)}
                                        className="flex justify-center gap-6"
                                    >
                                        <div className="flex items-center space-x-2">
                                            <RadioGroupItem value="Present" id={`${staff.id}-present`} />
                                            <Label htmlFor={`${staff.id}-present`}>Present</Label>
                                        </div>
                                        <div className="flex items-center space-x-2">
                                            <RadioGroupItem value="Absent" id={`${staff.id}-absent`} />
                                            <Label htmlFor={`${staff.id}-absent`}>Absent</Label>
                                        </div>
                                        <div className="flex items-center space-x-2">
                                            <RadioGroupItem value="Leave" id={`${staff.id}-leave`} />
                                            <Label htmlFor={`${staff.id}-leave`}>Leave</Label>
                                        </div>
                                    </RadioGroup>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
            <CardFooter className="border-t pt-6 justify-end">
                <Button onClick={handleSave}>
                    <Save className="mr-2 h-4 w-4" />
                    Save Attendance
                </Button>
            </CardFooter>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
