
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { ScanLine, Search, ArchiveRestore, CheckCircle, X } from 'lucide-react';
import { mockParcels } from '@/lib/data';
import { useToast } from '@/hooks/use-toast';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Checkbox } from '@/components/ui/checkbox';

type Bag = {
    id: string;
    sealId: string;
    parcels: Parcel[];
    origin: string;
};

const mockBags: Bag[] = [
    { id: 'BAG-123456', sealId: 'SEAL-1234', parcels: mockParcels.slice(0,3), origin: 'Lahore Hub'},
    { id: 'BAG-789012', sealId: 'SEAL-5678', parcels: mockParcels.slice(3,5), origin: 'Karachi Central'},
];

export default function AdminBranchUnbaggingPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [bagIdInput, setBagIdInput] = useState('');
  const [scannedBag, setScannedBag] = useState<Bag | null>(null);
  const [selectedParcels, setSelectedParcels] = useState<string[]>([]);
  const { toast } = useToast();

  const handleScanBag = () => {
    if (!bagIdInput) return;
    const bag = mockBags.find(b => b.id.toLowerCase() === bagIdInput.toLowerCase());
    
    if (bag) {
        setScannedBag(bag);
        setSelectedParcels(bag.parcels.map(p => p.id));
        toast({
            title: 'Bag Scanned',
            description: `${bag.parcels.length} parcels found in ${bag.id} from ${bag.origin}.`,
        });
    } else {
        setScannedBag(null);
        toast({
            variant: 'destructive',
            title: 'Scan Failed',
            description: 'No bag found with this ID.',
        });
    }
  };

  const handleSelectRow = (parcelId: string, checked: boolean) => {
    if (checked) {
      setSelectedParcels(prev => [...prev, parcelId]);
    } else {
      setSelectedParcels(prev => prev.filter(id => id !== parcelId));
    }
  };

  const handleProcess = () => {
    if (!scannedBag || selectedParcels.length === 0) return;
    toast({
        title: 'Unbagging Complete',
        description: `${selectedParcels.length} parcels have been processed and added to branch inventory.`
    });
    setScannedBag(null);
    setBagIdInput('');
    setSelectedParcels([]);
  }

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="branch-operator">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Unbagging - Scan Incoming Bag</CardTitle>
              <CardDescription>
                Scan the Bag ID to view its contents and process incoming shipments.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex items-center gap-2 max-w-lg mx-auto">
                    <div className="relative flex-1">
                        <ScanLine className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                        <Input 
                            placeholder="Scan or enter Bag ID..." 
                            className="pl-10 h-12 text-lg" 
                            value={bagIdInput}
                            onChange={(e) => setBagIdInput(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleScanBag()}
                        />
                    </div>
                    <Button size="lg" className="h-12" onClick={handleScanBag}>
                        <Search className="mr-2 h-5 w-5" /> Find Bag
                    </Button>
                </div>
            </CardContent>
          </Card>
           {scannedBag && (
            <Card className="animate-in fade-in-50">
                <CardHeader>
                    <CardTitle>Parcels in {scannedBag.id}</CardTitle>
                    <CardDescription>From: {scannedBag.origin} | Total Parcels: {scannedBag.parcels.length}</CardDescription>
                </CardHeader>
                <CardContent>
                     <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Parcel ID</TableHead>
                                <TableHead>Recipient</TableHead>
                                <TableHead>Destination City</TableHead>
                                <TableHead>COD Amount</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {scannedBag.parcels.map(p => (
                                <TableRow key={p.id}>
                                    <TableCell>{p.id}</TableCell>
                                    <TableCell>{p.recipientName}</TableCell>
                                    <TableCell>{p.city}</TableCell>
                                    <TableCell>PKR {p.codAmount.toFixed(2)}</TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                    </Table>
                </CardContent>
                 <CardFooter>
                    <Button onClick={handleProcess}>
                        <ArchiveRestore className="mr-2 h-4 w-4" />
                        Confirm Unbagging & Receive All Parcels
                    </Button>
                </CardFooter>
            </Card>
          )}
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
