
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow, TableFooter } from '@/components/ui/table';
import { PlusCircle, FileText } from 'lucide-react';
import { format } from 'date-fns';

type JournalEntry = {
  id: string;
  date: string;
  account: string;
  accountCode: string;
  debit: number;
  credit: number;
};

type JournalTransaction = {
  transactionId: string;
  date: string;
  description: string;
  entries: JournalEntry[];
};

const mockJournal: JournalTransaction[] = [
  {
    transactionId: 'JV-001',
    date: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(),
    description: 'Record daily COD collection',
    entries: [
      { id: 'e1', date: '', account: 'Cash and Bank', accountCode: '1010', debit: 45000, credit: 0 },
      { id: 'e2', date: '', account: 'Accounts Receivable - Merchants', accountCode: '1020', debit: 0, credit: 45000 },
    ],
  },
  {
    transactionId: 'JV-002',
    date: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString(),
    description: 'Pay monthly salaries',
    entries: [
      { id: 'e3', date: '', account: 'Salaries and Wages', accountCode: '5010', debit: 150000, credit: 0 },
      { id: 'e4', date: '', account: 'Cash and Bank', accountCode: '1010', debit: 0, credit: 150000 },
    ],
  },
];


export default function JournalEntriesPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader className="flex flex-row items-center">
              <div className="grid gap-2 flex-1">
                <CardTitle>Journal Entries</CardTitle>
                <CardDescription>
                  A chronological record of all financial transactions.
                </CardDescription>
              </div>
              <Button size="sm" className="ml-auto gap-1">
                <PlusCircle className="h-4 w-4" />
                Add Manual Entry
              </Button>
            </CardHeader>
            <CardContent>
                <div className="space-y-8">
                    {mockJournal.map(transaction => (
                        <Card key={transaction.transactionId} className="bg-muted/30">
                            <CardHeader className="flex-row items-center justify-between pb-4">
                                <div>
                                    <p className="font-semibold text-primary">{transaction.transactionId}</p>
                                    <p className="text-sm text-muted-foreground">{transaction.description}</p>
                                </div>
                                <p className="text-sm font-medium">{format(new Date(transaction.date), 'dd MMM, yyyy')}</p>
                            </CardHeader>
                            <CardContent>
                                <Table>
                                    <TableHeader>
                                        <TableRow>
                                            <TableHead>Account</TableHead>
                                            <TableHead className="text-right">Debit</TableHead>
                                            <TableHead className="text-right">Credit</TableHead>
                                        </TableRow>
                                    </TableHeader>
                                    <TableBody>
                                        {transaction.entries.map(entry => (
                                            <TableRow key={entry.id}>
                                                <TableCell>
                                                    <p className="font-medium">{entry.account}</p>
                                                    <p className="text-xs text-muted-foreground font-mono">{entry.accountCode}</p>
                                                </TableCell>
                                                <TableCell className="text-right font-mono">
                                                    {entry.debit > 0 ? `PKR ${entry.debit.toLocaleString()}` : '-'}
                                                </TableCell>
                                                <TableCell className="text-right font-mono">
                                                     {entry.credit > 0 ? `PKR ${entry.credit.toLocaleString()}` : '-'}
                                                </TableCell>
                                            </TableRow>
                                        ))}
                                    </TableBody>
                                    <TableFooter>
                                        <TableRow>
                                            <TableCell className="font-bold">Total</TableCell>
                                            <TableCell className="text-right font-bold font-mono">
                                                PKR {transaction.entries.reduce((sum, e) => sum + e.debit, 0).toLocaleString()}
                                            </TableCell>
                                            <TableCell className="text-right font-bold font-mono">
                                                PKR {transaction.entries.reduce((sum, e) => sum + e.credit, 0).toLocaleString()}
                                            </TableCell>
                                        </TableRow>
                                    </TableFooter>
                                </Table>
                            </CardContent>
                        </Card>
                    ))}
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
