
'use client';

import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { AlertTriangle, FileWarning, Banknote, MoreVertical } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { formatDistanceToNow } from 'date-fns';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

type AlertType = 'COD Variance' | 'Overdue Payment' | 'Large Expense';
type AlertPriority = 'High' | 'Medium' | 'Low';

type SystemAlert = {
  id: string;
  type: AlertType;
  title: string;
  description: string;
  timestamp: Date;
  priority: AlertPriority;
  status: 'New' | 'Investigating' | 'Resolved';
};

const alerts: SystemAlert[] = [
  {
    id: 'alert-1',
    type: 'COD Variance',
    title: 'COD Variance Detected - Multan Branch',
    description: 'A variance of PKR 5,000 has been detected between cash collected and deposited at the Multan Branch.',
    timestamp: new Date(Date.now() - 4 * 60 * 60 * 1000),
    priority: 'High',
    status: 'Investigating',
  },
  {
    id: 'alert-2',
    type: 'Overdue Payment',
    title: 'Vendor Payment Overdue: National Fuel Services',
    description: 'Invoice #INV-NFS-004 for PKR 80,000 is 5 days overdue.',
    timestamp: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000),
    priority: 'Medium',
    status: 'New',
  },
  {
    id: 'alert-3',
    type: 'Large Expense',
    title: 'Large Expense Claim Submitted',
    description: 'An expense claim of PKR 150,000 for "Marketing Event" requires approval.',
    timestamp: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000),
    priority: 'Low',
    status: 'New',
  },
];

const priorityStyles: Record<AlertPriority, string> = {
  High: 'border-red-500 bg-red-50 dark:bg-red-900/20',
  Medium: 'border-yellow-500 bg-yellow-50 dark:bg-yellow-900/20',
  Low: 'border-gray-300',
};

const iconMap: Record<AlertType, React.ElementType> = {
    'COD Variance': AlertTriangle,
    'Overdue Payment': FileWarning,
    'Large Expense': Banknote,
}

export default function SystemAlertsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Financial System Alerts</CardTitle>
              <CardDescription>
                A real-time feed of critical system-generated alerts that require attention from the finance team.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <div className="grid gap-6">
                    {alerts.map((alert) => {
                        const Icon = iconMap[alert.type];
                        return (
                             <Card key={alert.id} className={priorityStyles[alert.priority]}>
                                <CardHeader className="flex flex-row items-start gap-4 space-y-0 pb-4">
                                    <div className="flex h-10 w-10 items-center justify-center rounded-full bg-muted">
                                        <Icon className="h-5 w-5 text-muted-foreground" />
                                    </div>
                                    <div className="flex-1">
                                        <div className="flex items-center justify-between">
                                            <CardTitle className="text-base">{alert.title}</CardTitle>
                                            <Badge variant={alert.priority === 'High' ? 'destructive' : 'secondary'}>
                                                {alert.priority} Priority
                                            </Badge>
                                        </div>
                                        <CardDescription className="mt-1">{alert.description}</CardDescription>
                                    </div>
                                    <div>
                                         <DropdownMenu>
                                            <DropdownMenuTrigger asChild>
                                                <Button size="icon" variant="ghost">
                                                    <MoreVertical className="h-4 w-4" />
                                                </Button>
                                            </DropdownMenuTrigger>
                                            <DropdownMenuContent>
                                                <DropdownMenuItem>Mark as Investigating</DropdownMenuItem>
                                                <DropdownMenuItem>Mark as Resolved</DropdownMenuItem>
                                                <DropdownMenuItem>View Related Transaction</DropdownMenuItem>
                                            </DropdownMenuContent>
                                        </DropdownMenu>
                                    </div>
                                </CardHeader>
                                <CardFooter className="text-xs text-muted-foreground justify-between">
                                    <span>{formatDistanceToNow(alert.timestamp, { addSuffix: true })}</span>
                                    <Badge variant="outline">{alert.status}</Badge>
                                </CardFooter>
                            </Card>
                        )
                    })}
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
