
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { FileText, CheckCircle, PlusCircle } from 'lucide-react';
import { Badge } from '@/components/ui/badge';
import { format } from 'date-fns';

type ApprovalStatus = 'Pending' | 'Manager Approved' | 'Finance Approved' | 'Paid';

type ExpenseClaim = {
  id: string;
  description: string;
  amount: number;
  submittedBy: { name: string, avatar: string };
  status: ApprovalStatus;
  date: string;
};

const initialClaims: ExpenseClaim[] = [
  { id: 'exp-101', description: 'Office supplies for Multan branch', amount: 5000, submittedBy: { name: 'Ahmed Hassan', avatar: 'https://picsum.photos/seed/staff1/100' }, status: 'Manager Approved', date: '2024-07-28' },
  { id: 'exp-102', description: 'Client lunch meeting', amount: 3500, submittedBy: { name: 'Aisha Bibi', avatar: 'https://picsum.photos/seed/staff2/100' }, status: 'Pending', date: '2024-07-29' },
  { id: 'exp-103', description: 'Emergency vehicle repair - Rider 3', amount: 8000, submittedBy: { name: 'Bilal Khan', avatar: 'https://picsum.photos/seed/staff3/100' }, status: 'Finance Approved', date: '2024-07-27' },
  { id: 'exp-104', description: 'Monthly software subscription', amount: 12000, submittedBy: { name: 'Finance Dept', avatar: 'https://picsum.photos/seed/dept1/100' }, status: 'Paid', date: '2024-07-25' },
];

const pipelineStages: ApprovalStatus[] = ['Pending', 'Manager Approved', 'Finance Approved', 'Paid'];

export default function ApprovalFlowPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [claims, setClaims] = useState(initialClaims);

  const getStatusBadge = (status: ApprovalStatus) => {
      switch(status) {
        case 'Pending': return 'bg-yellow-100 text-yellow-800';
        case 'Manager Approved': return 'bg-blue-100 text-blue-800';
        case 'Finance Approved': return 'bg-purple-100 text-purple-800';
        case 'Paid': return 'bg-green-100 text-green-800';
      }
  }

  const ClaimCard = ({ claim }: { claim: ExpenseClaim }) => (
    <Card className="mb-4 cursor-grab active:cursor-grabbing">
      <CardContent className="p-3">
        <div className="flex justify-between items-start">
            <p className="font-semibold text-primary">PKR {claim.amount.toLocaleString()}</p>
            <Badge variant="outline" className={getStatusBadge(claim.status)}>{claim.status}</Badge>
        </div>
        <p className="text-sm my-2">{claim.description}</p>
        <div className="flex items-center justify-between text-xs text-muted-foreground">
            <div className="flex items-center gap-2">
                <Avatar className="h-6 w-6">
                    <AvatarImage src={claim.submittedBy.avatar} />
                    <AvatarFallback>{claim.submittedBy.name.charAt(0)}</AvatarFallback>
                </Avatar>
                <span>{claim.submittedBy.name}</span>
            </div>
            <span>{format(new Date(claim.date), 'dd MMM')}</span>
        </div>
      </CardContent>
      <CardFooter className="p-2 border-t flex justify-end gap-2">
          <Button size="sm" variant="ghost"><FileText className="mr-2 h-4 w-4"/>View</Button>
          <Button size="sm"><CheckCircle className="mr-2 h-4 w-4"/>Approve</Button>
      </CardFooter>
    </Card>
  );

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader className="flex flex-row items-center">
                <div className="flex-1">
                    <CardTitle>Expense Approval Flow</CardTitle>
                    <CardDescription>
                    Track expense claims from submission to payment.
                    </CardDescription>
                </div>
                <Button size="sm" className="ml-auto gap-1">
                    <PlusCircle className="h-4 w-4" />
                    New Expense Claim
                </Button>
            </CardHeader>
            <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    {pipelineStages.map(stage => (
                        <div key={stage} className="bg-muted/50 rounded-lg p-4">
                            <h3 className="font-semibold mb-4 text-center">{stage}</h3>
                            <div className="space-y-4 min-h-[200px]">
                                {claims.filter(claim => claim.status === stage).map(claim => (
                                    <ClaimCard key={claim.id} claim={claim} />
                                ))}
                            </div>
                        </div>
                    ))}
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
