
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Search, Download, Filter, User, FileText, WalletCards } from 'lucide-react';
import { format, addDays } from 'date-fns';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import type { DateRange } from 'react-day-picker';
import { Input } from '@/components/ui/input';

type LogEntry = {
  id: string;
  timestamp: string;
  user: { name: string; id: string; };
  action: string;
  details: string;
  ipAddress: string;
};

const mockLogs: LogEntry[] = [
  { id: 'log-1', timestamp: new Date(Date.now() - 2 * 60 * 60 * 1000).toISOString(), user: { name: 'Finance Manager', id: 'fin-mgr-1' }, action: 'Payroll Run', details: 'Processed payroll for July 2024', ipAddress: '192.168.1.1' },
  { id: 'log-2', timestamp: new Date(Date.now() - 5 * 60 * 60 * 1000).toISOString(), user: { name: 'Accountant', id: 'acc-1' }, action: 'Voucher Created', details: 'Created Bank Payment Voucher BPV-00456', ipAddress: '10.0.0.5' },
  { id: 'log-3', timestamp: new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString(), user: { name: 'Admin', id: 'admin-1' }, action: 'Permission Change', details: 'Granted "Approve Expenses" to role "Finance"', ipAddress: '203.0.113.25' },
  { id: 'log-4', timestamp: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(), user: { name: 'Accountant', id: 'acc-1' }, action: 'Invoice Generated', details: 'Generated invoice INV-2024-07-001 for Chic Boutique', ipAddress: '10.0.0.5' },
];

export default function AuditLogsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const [dateRange, setDateRange] = useState<DateRange | undefined>({
    from: addDays(new Date(), -7),
    to: new Date(),
  });

  const getActionIcon = (action: string) => {
      if (action.includes('Payroll')) return <WalletCards className="h-4 w-4 text-muted-foreground" />;
      if (action.includes('Voucher')) return <FileText className="h-4 w-4 text-muted-foreground" />;
      if (action.includes('Permission')) return <User className="h-4 w-4 text-muted-foreground" />;
      return <FileText className="h-4 w-4 text-muted-foreground" />;
  }

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Financial Audit Logs</CardTitle>
              <CardDescription>
                An immutable, chronological log of all actions taken within the finance module.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
                <div className="p-4 border rounded-lg bg-muted/50">
                     <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 items-end">
                        <div>
                            <label className="text-sm font-medium">Date Range</label>
                            <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                        </div>
                        <div>
                            <label className="text-sm font-medium">User or User ID</label>
                            <Input placeholder="Search by user..." />
                        </div>
                        <div>
                            <label className="text-sm font-medium">Action Type</label>
                            <Input placeholder="e.g., 'Payroll Run'" />
                        </div>
                        <div className="flex gap-2">
                             <Button className="w-full"><Search className="mr-2 h-4 w-4" /> Search Logs</Button>
                             <Button variant="outline" className="w-full"><Download className="mr-2 h-4 w-4"/> Export</Button>
                        </div>
                    </div>
                </div>

                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead className="w-[200px]">Timestamp</TableHead>
                            <TableHead>User</TableHead>
                            <TableHead>Action</TableHead>
                            <TableHead>Details</TableHead>
                            <TableHead>IP Address</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {mockLogs.map(log => (
                            <TableRow key={log.id}>
                                <TableCell>{format(new Date(log.timestamp), 'dd MMM yyyy, hh:mm:ss a')}</TableCell>
                                <TableCell>
                                    <div className="font-medium">{log.user.name}</div>
                                    <div className="text-xs text-muted-foreground">{log.user.id}</div>
                                </TableCell>
                                <TableCell>
                                    <div className="flex items-center gap-2">
                                        {getActionIcon(log.action)}
                                        <span>{log.action}</span>
                                    </div>
                                </TableCell>
                                <TableCell>{log.details}</TableCell>
                                <TableCell className="font-mono text-xs">{log.ipAddress}</TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
