
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { MoreVertical, PlusCircle, CalendarClock, Banknote, Hourglass } from 'lucide-react';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { format, isWithinInterval, startOfWeek, endOfWeek } from 'date-fns';

type ChequeStatus = 'Pending' | 'Cleared' | 'Bounced' | 'Cancelled';

type Cheque = {
  id: string;
  chequeNumber: string;
  payee: string;
  amount: number;
  issueDate: string;
  dueDate: string;
  status: ChequeStatus;
};

const mockCheques: Cheque[] = [
  { id: 'chq-1', chequeNumber: '123456', payee: 'Punjab Packaging', amount: 25000, issueDate: '2024-07-10', dueDate: '2024-07-25', status: 'Cleared' },
  { id: 'chq-2', chequeNumber: '123457', payee: 'National Fuel Services', amount: 80000, issueDate: '2024-07-15', dueDate: '2024-08-01', status: 'Pending' },
  { id: 'chq-3', chequeNumber: '123458', payee: 'City Vehicle Repairs', amount: 15000, issueDate: '2024-07-20', dueDate: '2024-07-28', status: 'Pending' },
  { id: 'chq-4', chequeNumber: '123459', payee: 'Office Supplies Co.', amount: 5000, issueDate: '2024-06-25', dueDate: '2024-07-10', status: 'Bounced' },
];

export default function ChequeManagementPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  
  const { pendingCheques, dueThisWeek, totalPendingValue } = useMemo(() => {
    const pending = mockCheques.filter(c => c.status === 'Pending');
    const start = startOfWeek(new Date());
    const end = endOfWeek(new Date());
    const due = pending.filter(c => isWithinInterval(new Date(c.dueDate), { start, end }));
    const totalValue = pending.reduce((sum, c) => sum + c.amount, 0);

    return {
        pendingCheques: pending.length,
        dueThisWeek: due.length,
        totalPendingValue: totalValue,
    }
  }, []);

  const getStatusBadge = (status: ChequeStatus) => {
    switch(status) {
        case 'Pending': return 'bg-yellow-100 text-yellow-800';
        case 'Cleared': return 'bg-green-100 text-green-800';
        case 'Bounced': return 'bg-red-100 text-red-800';
        case 'Cancelled': return 'bg-gray-200 text-gray-800';
        default: return 'bg-secondary';
    }
  };


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
            <div className="grid gap-4 md:grid-cols-3">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Pending Cheques</CardTitle>
                        <Hourglass className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{pendingCheques}</div>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Cheques Due This Week</CardTitle>
                        <CalendarClock className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{dueThisWeek}</div>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Value of Pending Cheques</CardTitle>
                        <Banknote className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">PKR {totalPendingValue.toLocaleString()}</div>
                    </CardContent>
                </Card>
            </div>
          <Card>
            <CardHeader className="flex flex-row items-center">
              <div className="grid gap-2">
                <CardTitle>Cheque Management</CardTitle>
                <CardDescription>
                  Track and manage post-dated and current cheques for payments.
                </CardDescription>
              </div>
               <Button size="sm" className="ml-auto gap-1">
                    <PlusCircle className="h-4 w-4" />
                    Add New Cheque
                </Button>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                    <TableRow>
                        <TableHead>Cheque #</TableHead>
                        <TableHead>Payee</TableHead>
                        <TableHead>Due Date</TableHead>
                        <TableHead>Amount</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {mockCheques.map(cheque => (
                        <TableRow key={cheque.id}>
                            <TableCell className="font-mono">{cheque.chequeNumber}</TableCell>
                            <TableCell>{cheque.payee}</TableCell>
                            <TableCell>{format(new Date(cheque.dueDate), 'dd MMM, yyyy')}</TableCell>
                            <TableCell>PKR {cheque.amount.toLocaleString()}</TableCell>
                            <TableCell>
                                <Badge variant="outline" className={getStatusBadge(cheque.status)}>{cheque.status}</Badge>
                            </TableCell>
                            <TableCell className="text-right">
                                 <DropdownMenu>
                                    <DropdownMenuTrigger asChild>
                                        <Button size="icon" variant="ghost">
                                            <MoreVertical className="h-4 w-4" />
                                        </Button>
                                    </DropdownMenuTrigger>
                                    <DropdownMenuContent>
                                        <DropdownMenuItem disabled={cheque.status !== 'Pending'}>Mark as Cleared</DropdownMenuItem>
                                        <DropdownMenuItem disabled={cheque.status !== 'Pending'}>Mark as Bounced</DropdownMenuItem>
                                        <DropdownMenuItem disabled={cheque.status !== 'Pending'}>Cancel Cheque</DropdownMenuItem>
                                    </DropdownMenuContent>
                                </DropdownMenu>
                            </TableCell>
                        </TableRow>
                    ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
