
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Rider, Branch } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { DatePicker } from '@/components/ui/date-picker';
import { Download, Banknote, Users, Building } from 'lucide-react';
import { mockRiders, mockBranches } from '@/lib/data';
import { format, isSameDay } from 'date-fns';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';

type RiderCollection = {
    rider: Rider;
    amount: number;
    shipmentCount: number;
    status: 'Verified' | 'Pending';
    collectionTime: string;
};

type BranchCollection = {
    branch: Branch;
    amount: number;
    depositSlipId: string;
    status: 'Verified' | 'Pending Verification';
    collectionTime: string;
};


export default function DailyCashCollectionPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [selectedDate, setSelectedDate] = useState<Date | undefined>(new Date());

  const dailyData = useMemo(() => {
    if (!selectedDate) return { riderCollections: [], branchCollections: [] };
    
    // Mock data generation for the selected date
    const riderCollections: RiderCollection[] = mockRiders.map((rider, index) => ({
        rider,
        amount: (Math.random() * 10000) + 5000,
        shipmentCount: Math.floor(Math.random() * 10) + 5,
        status: index % 2 === 0 ? 'Verified' : 'Pending',
        collectionTime: format(new Date(selectedDate.setHours(18 + index, 30)), 'hh:mm a'),
    }));
    
    const branchCollections: BranchCollection[] = mockBranches.slice(0, 2).map((branch, index) => ({
        branch,
        amount: (Math.random() * 50000) + 20000,
        depositSlipId: `SLIP-${Date.now().toString().slice(-4)}-${index}`,
        status: index === 0 ? 'Verified' : 'Pending Verification',
        collectionTime: format(new Date(selectedDate.setHours(17 - index, 0)), 'hh:mm a'),
    }));

    return { riderCollections, branchCollections };

  }, [selectedDate]);

  const totalRiderCash = dailyData.riderCollections.reduce((sum, item) => sum + item.amount, 0);
  const totalBranchCash = dailyData.branchCollections.reduce((sum, item) => sum + item.amount, 0);

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
               <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                  <div>
                    <CardTitle>Daily Cash Collection Report</CardTitle>
                    <CardDescription>
                      Cash collected from riders and branches on {selectedDate ? format(selectedDate, 'PPP') : ''}.
                    </CardDescription>
                  </div>
                  <div className="flex items-center gap-4">
                      <DatePicker date={selectedDate} setDate={setSelectedDate} />
                      <Button variant="outline">
                        <Download className="mr-2 h-4 w-4" />
                        Export Report
                      </Button>
                  </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-8">
                <div className="grid gap-4 md:grid-cols-3">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Collected</CardTitle>
                            <Banknote className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {(totalRiderCash + totalBranchCash).toLocaleString()}</div>
                        </CardContent>
                    </Card>
                     <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">From Riders</CardTitle>
                            <Users className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {totalRiderCash.toLocaleString()}</div>
                        </CardContent>
                    </Card>
                     <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">From Branches</CardTitle>
                            <Building className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {totalBranchCash.toLocaleString()}</div>
                        </CardContent>
                    </Card>
                </div>

                <div>
                    <h3 className="text-lg font-semibold mb-4">Cash Handover from Riders</h3>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Rider</TableHead>
                                <TableHead>Shipments</TableHead>
                                <TableHead>Collection Time</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead className="text-right">Amount (PKR)</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {dailyData.riderCollections.map(item => (
                                <TableRow key={item.rider.id}>
                                    <TableCell>
                                        <div className="flex items-center gap-3">
                                            <Avatar>
                                                <AvatarImage src={item.rider.avatar} />
                                                <AvatarFallback>{item.rider.name.charAt(0)}</AvatarFallback>
                                            </Avatar>
                                            <p className="font-medium">{item.rider.name}</p>
                                        </div>
                                    </TableCell>
                                    <TableCell>{item.shipmentCount}</TableCell>
                                    <TableCell>{item.collectionTime}</TableCell>
                                    <TableCell>
                                        <Badge variant={item.status === 'Verified' ? 'default' : 'secondary'} className={item.status === 'Verified' ? 'bg-green-100 text-green-800' : ''}>
                                            {item.status}
                                        </Badge>
                                    </TableCell>
                                    <TableCell className="text-right font-medium">{item.amount.toLocaleString()}</TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                    </Table>
                </div>
                
                 <div>
                    <h3 className="text-lg font-semibold mb-4">Bulk Collection from Branches</h3>
                     <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Branch</TableHead>
                                <TableHead>Deposit Slip ID</TableHead>
                                <TableHead>Collection Time</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead className="text-right">Amount (PKR)</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                           {dailyData.branchCollections.map(item => (
                                <TableRow key={item.branch.id}>
                                    <TableCell>
                                        <div className="flex items-center gap-3">
                                            <Avatar>
                                                <AvatarImage src={item.branch.avatar} />
                                                <AvatarFallback>{item.branch.name?.charAt(0)}</AvatarFallback>
                                            </Avatar>
                                            <p className="font-medium">{item.branch.name}</p>
                                        </div>
                                    </TableCell>
                                    <TableCell>{item.depositSlipId}</TableCell>
                                    <TableCell>{item.collectionTime}</TableCell>
                                    <TableCell>
                                        <Badge variant={item.status === 'Verified' ? 'default' : 'destructive'} className={item.status === 'Verified' ? 'bg-green-100 text-green-800' : ''}>
                                            {item.status}
                                        </Badge>
                                    </TableCell>
                                    <TableCell className="text-right font-medium">{item.amount.toLocaleString()}</TableCell>
                                </TableRow>
                           ))}
                        </TableBody>
                    </Table>
                </div>

            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}

