
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Branch } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import type { DateRange } from 'react-day-picker';
import { Download, Upload, CheckCircle, Clock } from 'lucide-react';
import { mockBranches } from '@/lib/data';
import { addDays, format } from 'date-fns';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';

type Deposit = {
    id: string;
    branch: Branch;
    amount: number;
    depositDate: string;
    status: 'Verified' | 'Pending Verification';
    depositSlipId: string;
};

const mockDeposits: Deposit[] = [
    { id: 'DEP-001', branch: mockBranches[0], amount: 45000, depositDate: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString(), status: 'Verified', depositSlipId: 'SLIP-A1' },
    { id: 'DEP-002', branch: mockBranches[1], amount: 88000, depositDate: new Date().toISOString(), status: 'Pending Verification', depositSlipId: 'SLIP-B1' },
    { id: 'DEP-003', branch: mockBranches[2], amount: 72000, depositDate: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(), status: 'Verified', depositSlipId: 'SLIP-C1' },
];

export default function DepositsToBankPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [dateRange, setDateRange] = useState<DateRange | undefined>({
    from: addDays(new Date(), -7),
    to: new Date(),
  });
  
  const filteredDeposits = useMemo(() => {
    return mockDeposits.filter(d => {
        const depositDate = new Date(d.depositDate);
        return dateRange?.from && dateRange?.to && depositDate >= dateRange.from && depositDate <= dateRange.to;
    });
  }, [dateRange]);

  const totalDeposited = filteredDeposits.reduce((sum, d) => sum + d.amount, 0);
  const pendingVerification = filteredDeposits.filter(d => d.status === 'Pending Verification').length;


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
               <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                  <div>
                    <CardTitle>Bank Deposit Slips</CardTitle>
                    <CardDescription>
                      Upload, verify, and track all bank deposit slips from branches.
                    </CardDescription>
                  </div>
                  <div className="flex items-center gap-4">
                      <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                      <Button>
                        <Upload className="mr-2 h-4 w-4" />
                        Upload Slip
                      </Button>
                  </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-8">
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Deposits</CardTitle>
                            <Download className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {totalDeposited.toLocaleString()}</div>
                        </CardContent>
                    </Card>
                     <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Pending Verification</CardTitle>
                            <Clock className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{pendingVerification}</div>
                        </CardContent>
                    </Card>
                </div>

                <div>
                    <h3 className="text-lg font-semibold mb-4">Deposit History</h3>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Branch</TableHead>
                                <TableHead>Deposit Slip ID</TableHead>
                                <TableHead>Deposit Date</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead className="text-right">Amount (PKR)</TableHead>
                                <TableHead className="text-right">Action</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {filteredDeposits.map(item => (
                                <TableRow key={item.id}>
                                    <TableCell>
                                        <div className="flex items-center gap-3">
                                            <Avatar>
                                                <AvatarImage src={item.branch.avatar} />
                                                <AvatarFallback>{item.branch.name?.charAt(0)}</AvatarFallback>
                                            </Avatar>
                                            <p className="font-medium">{item.branch.name}</p>
                                        </div>
                                    </TableCell>
                                    <TableCell>{item.depositSlipId}</TableCell>
                                    <TableCell>{format(new Date(item.depositDate), 'dd MMM, yyyy')}</TableCell>
                                    <TableCell>
                                        <Badge variant={item.status === 'Verified' ? 'default' : 'destructive'} className={item.status === 'Verified' ? 'bg-green-100 text-green-800' : ''}>
                                            {item.status}
                                        </Badge>
                                    </TableCell>
                                    <TableCell className="text-right font-medium">{item.amount.toLocaleString()}</TableCell>
                                    <TableCell className="text-right">
                                        <Button size="sm" variant="outline" disabled={item.status === 'Verified'}>
                                            <CheckCircle className="mr-2 h-4 w-4" />
                                            Verify
                                        </Button>
                                    </TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                    </Table>
                </div>
            </CardContent>
             <CardFooter>
                <Button variant="outline">
                    <Download className="mr-2 h-4 w-4" />
                    Export Report
                </Button>
            </CardFooter>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
