
'use client';

import { useState, useEffect } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Banknote, Receipt, Hourglass, TrendingUp, TrendingDown, Bell, FileText, CheckCircle, Clock, Truck, FileWarning, BookUser, FileClock, Check } from 'lucide-react';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import { type DateRange } from 'react-day-picker';
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Button } from '@/components/ui/button';
import { format } from 'date-fns';



export default function AdminFinanceDashboardPage() {
  const [data, setData] = useState<any>(null);
  const [dateRange, setDateRange] = useState<DateRange | undefined>();

  useEffect(() => {
    const fetchData = async () => {
      try {
        const response = await fetch('/api/finance/dashboard-data');
        const result = await response.json();
        setData(result);
      } catch (error) {
        console.error('Error fetching dashboard data:', error);
      }
    };

    fetchData();
  }, []);

  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
            
            <Card>
                <CardHeader>
                    <CardTitle>Filters</CardTitle>
                </CardHeader>
                <CardContent>
                   <div className="flex flex-col md:flex-row gap-4">
                        <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                        <Select>
                            <SelectTrigger>
                                <SelectValue placeholder="Select Agent" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="agent1">Agent 1</SelectItem>
                                <SelectItem value="agent2">Agent 2</SelectItem>
                            </SelectContent>
                        </Select>
                   </div>
                </CardContent>
            </Card>

            <Card>
                <CardHeader>
                    <CardTitle>Overview KPIs</CardTitle>
                </CardHeader>
                <CardContent className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">TOTAL DELIVERIES</CardTitle>
                            <Truck className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{data?.stats?.total_deliveries ?? '...'}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">COD COLLECTED</CardTitle>
                            <Banknote className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {data?.stats?.cod_collected?.toLocaleString() ?? '...'}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">DEPOSITS</CardTitle>
                            <Receipt className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {data?.stats?.deposits?.toLocaleString() ?? '...'}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">VARIANCE</CardTitle>
                            <FileWarning className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-red-600">PKR {data?.stats?.variance?.toLocaleString() ?? '...'}</div>
                        </CardContent>
                    </Card>
                </CardContent>
            </Card>

            <div className="grid gap-8 md:grid-cols-2">
                <Card>
                    <CardHeader>
                        <CardTitle>Agent Wallets</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Agent</TableHead>
                                    <TableHead>Wallet Balance</TableHead>
                                    <TableHead>Cleared</TableHead>
                                    <TableHead>Actions</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {data?.agentWallets?.map((wallet: any) => (
                                    <TableRow key={wallet.agent}>
                                        <TableCell>{wallet.agent}</TableCell>
                                        <TableCell>PKR {wallet.balance.toLocaleString()}</TableCell>
                                        <TableCell>{wallet.cleared ? <Check className="h-5 w-5 text-green-500" /> : 'Not'}</TableCell>
                                        <TableCell>
                                            {wallet.cleared ?
                                                <Button variant="link" className="p-0 h-auto">Mark Deposited</Button>
                                                : wallet.action
                                            }
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle>Tamper-Proof Audit Trail</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <ul className="space-y-3">
                            {data?.auditTrail?.map((item: any, index: number) => (
                                <li key={index} className="flex justify-between items-center text-sm">
                                    <span>{item.activity} <span className="font-semibold">PKR {item.amount}</span></span>
                                    <span className="text-muted-foreground">{item.date}</span>
                                </li>
                            ))}
                        </ul>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle>Ledgers</CardTitle>
                    </CardHeader>
                    <CardContent>
                         <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Date</TableHead>
                                    <TableHead>Agent</TableHead>
                                    <TableHead>Amount</TableHead>
                                    <TableHead>Status</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {data?.ledgers?.map((ledger: any) => (
                                    <TableRow key={ledger.date + ledger.agent}>
                                        <TableCell>{format(new Date(ledger.date), 'dd-MM-yyyy')}</TableCell>
                                        <TableCell>{ledger.agent}</TableCell>
                                        <TableCell>PKR {ledger.amount.toLocaleString()}</TableCell>
                                        <TableCell>{ledger.status}</TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle>Audit Trail</CardTitle>
                    </CardHeader>
                    <CardContent>
                         <ul className="space-y-3">
                            {data?.auditTrail?.slice(0, 2).map((item: any, index: number) => (
                                <li key={index} className="flex justify-between items-center text-sm">
                                    <span>{item.activity} <span className="font-semibold">PKR {item.amount}</span></span>
                                    <span className="text-muted-foreground">{item.date}</span>
                                </li>
                            ))}
                        </ul>
                    </CardContent>
                </Card>
            </div>
            
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}

    