
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem } from '@/components/ui/dropdown-menu';
import { MoreVertical, PlusCircle, Receipt, Clock, CheckCircle } from 'lucide-react';
import { format } from 'date-fns';

type Expense = {
  id: string;
  date: string;
  category: 'Fuel' | 'Utilities' | 'Rent' | 'Salaries' | 'Maintenance';
  description: string;
  amount: number;
  status: 'Pending' | 'Approved' | 'Paid';
};

const mockExpenses: Expense[] = [
  { id: 'exp-1', date: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(), category: 'Fuel', description: 'Fuel for delivery fleet - Multan Branch', amount: 15000, status: 'Paid' },
  { id: 'exp-2', date: new Date(Date.now() - 5 * 24 * 60 * 60 * 1000).toISOString(), category: 'Rent', description: 'Warehouse Rent - Lahore Hub', amount: 150000, status: 'Paid' },
  { id: 'exp-3', date: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString(), category: 'Utilities', description: 'Electricity Bill - Karachi Central', amount: 25000, status: 'Approved' },
  { id: 'exp-4', date: new Date().toISOString(), category: 'Maintenance', description: 'Vehicle repair for Rider-1', amount: 5000, status: 'Pending' },
];


export default function FinanceExpensesPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const totalThisMonth = mockExpenses.reduce((sum, exp) => sum + exp.amount, 0);
  const pendingApproval = mockExpenses.filter(e => e.status === 'Pending').length;
  const totalPaid = mockExpenses.filter(e => e.status === 'Paid').reduce((sum, exp) => sum + exp.amount, 0);

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Expenses (This Month)</CardTitle>
                        <Receipt className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">PKR {totalThisMonth.toLocaleString()}</div>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Pending Approval</CardTitle>
                        <Clock className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{pendingApproval}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Total Paid</CardTitle>
                        <CheckCircle className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">PKR {totalPaid.toLocaleString()}</div>
                    </CardContent>
                </Card>
            </div>
          <Card>
            <CardHeader className="flex flex-row items-center">
                <div className="grid gap-2">
                    <CardTitle>Expense Tracking</CardTitle>
                    <CardDescription>
                        Track and manage all business expenses.
                    </CardDescription>
                </div>
                <Button size="sm" className="ml-auto gap-1">
                    <PlusCircle className="h-4 w-4" />
                    Add Expense
                </Button>
            </CardHeader>
            <CardContent>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Date</TableHead>
                            <TableHead>Category</TableHead>
                            <TableHead>Description</TableHead>
                            <TableHead>Amount</TableHead>
                            <TableHead>Status</TableHead>
                            <TableHead className="text-right">Actions</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {mockExpenses.map(expense => (
                            <TableRow key={expense.id}>
                                <TableCell>{format(new Date(expense.date), 'dd MMM, yyyy')}</TableCell>
                                <TableCell><Badge variant="secondary">{expense.category}</Badge></TableCell>
                                <TableCell>{expense.description}</TableCell>
                                <TableCell>PKR {expense.amount.toLocaleString()}</TableCell>
                                <TableCell>
                                    <Badge variant={
                                        expense.status === 'Paid' ? 'default' : 
                                        expense.status === 'Approved' ? 'secondary' : 'destructive'
                                    }>{expense.status}</Badge>
                                </TableCell>
                                <TableCell className="text-right">
                                     <DropdownMenu>
                                        <DropdownMenuTrigger asChild>
                                            <Button size="icon" variant="ghost">
                                                <MoreVertical className="h-4 w-4" />
                                            </Button>
                                        </DropdownMenuTrigger>
                                        <DropdownMenuContent>
                                            <DropdownMenuItem>View Details</DropdownMenuItem>
                                            <DropdownMenuItem disabled={expense.status !== 'Pending'}>Approve</DropdownMenuItem>
                                            <DropdownMenuItem disabled={expense.status !== 'Approved'}>Mark as Paid</DropdownMenuItem>
                                        </DropdownMenuContent>
                                    </DropdownMenu>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
