
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem } from '@/components/ui/dropdown-menu';
import { MoreVertical, PlusCircle, Repeat, ToggleLeft, ToggleRight } from 'lucide-react';
import { format } from 'date-fns';

type RecurringExpense = {
  id: string;
  description: string;
  category: string;
  amount: number;
  frequency: 'Monthly' | 'Quarterly' | 'Yearly';
  nextDueDate: string;
  status: 'Active' | 'Inactive';
};

const mockRecurringExpenses: RecurringExpense[] = [
  { id: 'rec-1', description: 'Warehouse Rent - Lahore Hub', category: 'Rent', amount: 150000, frequency: 'Monthly', nextDueDate: '2024-08-01', status: 'Active' },
  { id: 'rec-2', description: 'Shopify Subscription', category: 'Software', amount: 12000, frequency: 'Monthly', nextDueDate: '2024-08-15', status: 'Active' },
  { id: 'rec-3', description: 'Vehicle Insurance', category: 'Insurance', amount: 80000, frequency: 'Yearly', nextDueDate: '2025-01-01', status: 'Active' },
  { id: 'rec-4', description: 'Quarterly Audit Fees', category: 'Professional Fees', amount: 50000, frequency: 'Quarterly', nextDueDate: '2024-10-01', status: 'Inactive' },
];

export default function RecurringExpensesPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader className="flex flex-row items-center">
                <div className="grid gap-2">
                    <CardTitle>Recurring Expenses</CardTitle>
                    <CardDescription>
                        Automate and manage recurring payments like rent, utilities, and subscriptions.
                    </CardDescription>
                </div>
                <Button size="sm" className="ml-auto gap-1">
                    <PlusCircle className="h-4 w-4" />
                    Add Recurring Expense
                </Button>
            </CardHeader>
            <CardContent>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Description</TableHead>
                            <TableHead>Category</TableHead>
                            <TableHead>Frequency</TableHead>
                            <TableHead>Next Due Date</TableHead>
                            <TableHead>Amount</TableHead>
                            <TableHead>Status</TableHead>
                            <TableHead className="text-right">Actions</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {mockRecurringExpenses.map(expense => (
                            <TableRow key={expense.id}>
                                <TableCell className="font-medium">{expense.description}</TableCell>
                                <TableCell>{expense.category}</TableCell>
                                <TableCell>{expense.frequency}</TableCell>
                                <TableCell>{format(new Date(expense.nextDueDate), 'dd MMM, yyyy')}</TableCell>
                                <TableCell>PKR {expense.amount.toLocaleString()}</TableCell>
                                <TableCell>
                                    <Badge variant={expense.status === 'Active' ? 'default' : 'secondary'} className={expense.status === 'Active' ? 'bg-green-100 text-green-800' : ''}>
                                        {expense.status}
                                    </Badge>
                                </TableCell>
                                <TableCell className="text-right">
                                     <DropdownMenu>
                                        <DropdownMenuTrigger asChild>
                                            <Button size="icon" variant="ghost">
                                                <MoreVertical className="h-4 w-4" />
                                            </Button>
                                        </DropdownMenuTrigger>
                                        <DropdownMenuContent>
                                            <DropdownMenuItem>Edit</DropdownMenuItem>
                                            <DropdownMenuItem>
                                                {expense.status === 'Active' ? <ToggleLeft className="mr-2 h-4 w-4"/> : <ToggleRight className="mr-2 h-4 w-4"/>}
                                                {expense.status === 'Active' ? 'Deactivate' : 'Activate'}
                                            </DropdownMenuItem>
                                            <DropdownMenuItem>View History</DropdownMenuItem>
                                        </DropdownMenuContent>
                                    </DropdownMenu>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
