
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { CalendarDays, Check, X, Plane, Clock } from 'lucide-react';
import { eachDayOfInterval, format, startOfMonth, endOfMonth, getDay } from 'date-fns';

const mockEmployeeData = [
    { id: 'emp-1', name: 'Ahmed Hassan' },
    { id: 'emp-2', name: 'Aisha Bibi' },
    { id: 'emp-3', name: 'Bilal Khan' },
];

const mockAttendance = {
    'emp-1': { '2024-07-03': 'Present', '2024-07-04': 'Present', '2024-07-10': 'Leave', '2024-07-15': 'Absent' },
    'emp-2': { '2024-07-01': 'Present', '2024-07-02': 'Present', '2024-07-08': 'Present', '2024-07-22': 'Present' },
};


export default function AttendancePage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [selectedMonth, setSelectedMonth] = useState(format(new Date(), 'yyyy-MM'));
  const [selectedEmployee, setSelectedEmployee] = useState('emp-1');

  const { daysInMonth, summary } = useMemo(() => {
    const monthStart = startOfMonth(new Date(selectedMonth));
    const monthEnd = endOfMonth(monthStart);
    const days = eachDayOfInterval({ start: monthStart, end: monthEnd });
    
    const employeeAttendance = mockAttendance[selectedEmployee as keyof typeof mockAttendance] || {};

    const summaryData = days.reduce((acc, day) => {
        const dayKey = format(day, 'yyyy-MM-dd');
        const status = employeeAttendance[dayKey as keyof typeof employeeAttendance] || (getDay(day) > 0 && getDay(day) < 6 ? 'N/A' : 'Weekend');
        if (status === 'Present') acc.present++;
        if (status === 'Absent') acc.absent++;
        if (status === 'Leave') acc.leave++;
        return acc;
    }, { present: 0, absent: 0, leave: 0, overtime: Math.floor(Math.random() * 10) });

    return { daysInMonth: days, summary: summaryData };
  }, [selectedMonth, selectedEmployee]);

  const getStatusCell = (day: Date) => {
      const dayKey = format(day, 'yyyy-MM-dd');
      const employeeAttendance = mockAttendance[selectedEmployee as keyof typeof mockAttendance] || {};
      const status = employeeAttendance[dayKey as keyof typeof employeeAttendance] || 'N/A';
      const isWeekend = getDay(day) === 0 || getDay(day) === 6;

      if (isWeekend) return <TableCell className="text-center text-muted-foreground bg-muted/50">Weekend</TableCell>;
      switch(status) {
          case 'Present': return <TableCell className="text-center bg-green-50 text-green-700 font-semibold">P</TableCell>;
          case 'Absent': return <TableCell className="text-center bg-red-50 text-red-700 font-semibold">A</TableCell>;
          case 'Leave': return <TableCell className="text-center bg-yellow-50 text-yellow-700 font-semibold">L</TableCell>;
          default: return <TableCell className="text-center text-muted-foreground">N/A</TableCell>;
      }
  }


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Attendance & Overtime</CardTitle>
              <CardDescription>
                Review employee attendance records for payroll processing.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-8">
                 <div className="p-4 border rounded-lg bg-muted/50 flex flex-col md:flex-row gap-4">
                    <div className="flex-1">
                        <label className="text-sm font-medium">Select Month</label>
                        <Input type="month" value={selectedMonth} onChange={(e) => setSelectedMonth(e.target.value)} />
                    </div>
                    <div className="flex-1">
                        <label className="text-sm font-medium">Select Employee</label>
                        <Select value={selectedEmployee} onValueChange={setSelectedEmployee}>
                            <SelectTrigger><SelectValue/></SelectTrigger>
                            <SelectContent>
                                {mockEmployeeData.map(emp => <SelectItem key={emp.id} value={emp.id}>{emp.name}</SelectItem>)}
                            </SelectContent>
                        </Select>
                    </div>
                </div>

                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Present</CardTitle>
                        <Check className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent><div className="text-2xl font-bold">{summary.present}</div></CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Absent</CardTitle>
                        <X className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent><div className="text-2xl font-bold">{summary.absent}</div></CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">On Leave</CardTitle>
                        <Plane className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent><div className="text-2xl font-bold">{summary.leave}</div></CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Overtime (Hours)</CardTitle>
                        <Clock className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent><div className="text-2xl font-bold">{summary.overtime}</div></CardContent>
                    </Card>
                </div>

                <div>
                    <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                        <CalendarDays className="h-5 w-5"/>
                        {format(new Date(selectedMonth), 'MMMM yyyy')} Attendance
                    </h3>
                    <div className="border rounded-lg">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    {daysInMonth.map(day => <TableHead key={day.toString()} className="text-center">{format(day, 'd')}</TableHead>)}
                                </TableRow>
                            </TableHeader>
                             <TableBody>
                                <TableRow>
                                    {daysInMonth.map(day => getStatusCell(day))}
                                </TableRow>
                            </TableBody>
                        </Table>
                    </div>
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
