
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { PlayCircle, Users, Calendar, Banknote } from 'lucide-react';
import { format } from 'date-fns';
import { Input } from '@/components/ui/input';
import { useToast } from '@/hooks/use-toast';

type PayrollEntry = {
    employeeId: string;
    name: string;
    grossSalary: number;
    deductions: number;
    netPay: number;
};

const mockPayrollData: PayrollEntry[] = [
    { employeeId: 'emp-1', name: 'Ahmed Hassan', grossSalary: 52000, deductions: 4000, netPay: 48000 },
    { employeeId: 'emp-2', name: 'Aisha Bibi', grossSalary: 50000, deductions: 3500, netPay: 46500 },
    { employeeId: 'emp-3', name: 'Bilal Khan', grossSalary: 37000, deductions: 1500, netPay: 35500 },
];

export default function PayrollRunPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [selectedMonth, setSelectedMonth] = useState(format(new Date(), 'yyyy-MM'));
  const [payrollRun, setPayrollRun] = useState<PayrollEntry[] | null>(null);
  const { toast } = useToast();

  const handleRunPayroll = () => {
    setPayrollRun(mockPayrollData);
    toast({ title: "Payroll Processed", description: `Payroll for ${format(new Date(selectedMonth), 'MMMM yyyy')} has been calculated.`});
  };

  const summary = useMemo(() => {
      if (!payrollRun) return { totalCost: 0, totalEmployees: 0 };
      return {
          totalCost: payrollRun.reduce((sum, p) => sum + p.netPay, 0),
          totalEmployees: payrollRun.length,
      }
  }, [payrollRun]);

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Payroll Run</CardTitle>
              <CardDescription>
                Calculate and process monthly salaries for all employees.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-8">
                <div className="p-4 border rounded-lg bg-muted/50 flex flex-col md:flex-row gap-4 items-end">
                    <div className="flex-1">
                        <label className="text-sm font-medium">Select Month for Payroll</label>
                        <Input type="month" value={selectedMonth} onChange={(e) => setSelectedMonth(e.target.value)} />
                    </div>
                    <Button onClick={handleRunPayroll} className="w-full md:w-auto">
                        <PlayCircle className="mr-2 h-4 w-4" /> Run Payroll
                    </Button>
                </div>

                {payrollRun && (
                    <div className="animate-in fade-in-50">
                         <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3 mb-8">
                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium">Total Payroll Cost</CardTitle>
                                <Banknote className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent><div className="text-2xl font-bold">PKR {summary.totalCost.toLocaleString()}</div></CardContent>
                            </Card>
                             <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium">Total Employees</CardTitle>
                                <Users className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent><div className="text-2xl font-bold">{summary.totalEmployees}</div></CardContent>
                            </Card>
                             <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium">Pay Date</CardTitle>
                                <Calendar className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent><div className="text-2xl font-bold">{format(new Date(), 'dd MMM, yyyy')}</div></CardContent>
                            </Card>
                        </div>
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Employee</TableHead>
                                    <TableHead>Gross Salary</TableHead>
                                    <TableHead>Deductions</TableHead>
                                    <TableHead>Net Pay</TableHead>
                                    <TableHead className="text-right">Action</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {payrollRun.map(item => (
                                    <TableRow key={item.employeeId}>
                                        <TableCell className="font-medium">{item.name}</TableCell>
                                        <TableCell>PKR {item.grossSalary.toLocaleString()}</TableCell>
                                        <TableCell>PKR {item.deductions.toLocaleString()}</TableCell>
                                        <TableCell className="font-bold">PKR {item.netPay.toLocaleString()}</TableCell>
                                        <TableCell className="text-right">
                                            <Button variant="outline" size="sm">View Payslip</Button>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </div>
                )}
            </CardContent>
             {payrollRun && (
                <CardFooter className="border-t pt-6 justify-end">
                    <Button>Approve & Finalize Payroll</Button>
                </CardFooter>
            )}
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
