
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Download, FileText } from 'lucide-react';
import { format } from 'date-fns';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

type SalarySlip = {
    id: string;
    employeeName: string;
    month: string;
    netSalary: number;
};

const mockSlips: SalarySlip[] = [
    { id: 'slip-1', employeeName: 'Ahmed Hassan', month: '2024-07', netSalary: 48000 },
    { id: 'slip-2', employeeName: 'Aisha Bibi', month: '2024-07', netSalary: 46500 },
    { id: 'slip-3', employeeName: 'Bilal Khan', month: '2024-07', netSalary: 35500 },
    { id: 'slip-4', employeeName: 'Ahmed Hassan', month: '2024-06', netSalary: 47500 },
];

export default function SalarySlipsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [selectedMonth, setSelectedMonth] = useState(format(new Date(), 'yyyy-MM'));
  const [searchTerm, setSearchTerm] = useState('');

  const filteredSlips = useMemo(() => {
    return mockSlips.filter(slip => 
        slip.month === selectedMonth &&
        slip.employeeName.toLowerCase().includes(searchTerm.toLowerCase())
    );
  }, [selectedMonth, searchTerm]);


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Generated Salary Slips</CardTitle>
              <CardDescription>
                View and download salary slips for all employees.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-8">
                 <div className="p-4 border rounded-lg bg-muted/50 flex flex-col md:flex-row gap-4">
                    <div className="flex-1">
                        <label className="text-sm font-medium">Select Month</label>
                        <Input type="month" value={selectedMonth} onChange={(e) => setSelectedMonth(e.target.value)} />
                    </div>
                     <div className="flex-1">
                        <label className="text-sm font-medium">Search Employee</label>
                        <Input placeholder="Enter employee name..." value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                    </div>
                </div>

                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Employee</TableHead>
                            <TableHead>Month</TableHead>
                            <TableHead>Net Salary</TableHead>
                            <TableHead className="text-right">Action</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {filteredSlips.map(slip => (
                             <TableRow key={slip.id}>
                                <TableCell className="font-medium">{slip.employeeName}</TableCell>
                                <TableCell>{format(new Date(slip.month), 'MMMM yyyy')}</TableCell>
                                <TableCell>PKR {slip.netSalary.toLocaleString()}</TableCell>
                                <TableCell className="text-right">
                                    <Button variant="outline" size="sm">
                                        <Download className="mr-2 h-4 w-4" /> Download
                                    </Button>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
