
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { MoreVertical, PlusCircle, Calculator, UserSearch } from 'lucide-react';
import { Input } from '@/components/ui/input';

type TaxSlab = {
    id: string;
    minIncome: number;
    maxIncome: number | null;
    rate: number;
    fixedAmount: number;
};

const mockTaxSlabs: TaxSlab[] = [
    { id: 'slab-1', minIncome: 0, maxIncome: 600000, rate: 0, fixedAmount: 0 },
    { id: 'slab-2', minIncome: 600001, maxIncome: 1200000, rate: 2.5, fixedAmount: 0 },
    { id: 'slab-3', minIncome: 1200001, maxIncome: 2400000, rate: 12.5, fixedAmount: 15000 },
    { id: 'slab-4', minIncome: 2400001, maxIncome: 3600000, rate: 22.5, fixedAmount: 165000 },
];

export default function TaxDeductionsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [annualSalary, setAnnualSalary] = useState('');
  const [taxResult, setTaxResult] = useState<{ slab: TaxSlab; totalTax: number } | null>(null);

  const handleCalculateTax = () => {
    const salary = Number(annualSalary);
    if (isNaN(salary) || salary <= 0) return;

    const applicableSlab = mockTaxSlabs
        .slice()
        .reverse()
        .find(slab => salary >= slab.minIncome);

    if (applicableSlab) {
        const taxableAmount = salary - applicableSlab.minIncome;
        const variableTax = (taxableAmount * applicableSlab.rate) / 100;
        const totalTax = applicableSlab.fixedAmount + variableTax;
        setTaxResult({ slab: applicableSlab, totalTax });
    }
  };

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader className="flex flex-row items-center">
                <div className="grid gap-2 flex-1">
                    <CardTitle>Income Tax Slabs (2023-2024)</CardTitle>
                    <CardDescription>
                        Manage and review the tax slabs for salary deductions.
                    </CardDescription>
                </div>
                 <Button size="sm" className="ml-auto gap-1">
                    <PlusCircle className="h-4 w-4" />
                    Add New Slab
                </Button>
            </CardHeader>
            <CardContent>
                <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Income Range (PKR)</TableHead>
                            <TableHead>Tax Rate</TableHead>
                            <TableHead>Fixed Amount (PKR)</TableHead>
                            <TableHead className="text-right">Actions</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {mockTaxSlabs.map(slab => (
                            <TableRow key={slab.id}>
                                <TableCell className="font-medium">
                                    {slab.minIncome.toLocaleString()} - {slab.maxIncome ? slab.maxIncome.toLocaleString() : 'Above'}
                                </TableCell>
                                <TableCell>{slab.rate}%</TableCell>
                                <TableCell>{slab.fixedAmount.toLocaleString()}</TableCell>
                                <TableCell className="text-right">
                                    <Button variant="ghost" size="icon"><MoreVertical className="h-4 w-4" /></Button>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>

           <Card>
            <CardHeader>
                <CardTitle>Tax Calculator</CardTitle>
                <CardDescription>
                    Estimate the income tax for a given annual salary.
                </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
                <div className="flex flex-col md:flex-row gap-4 items-end max-w-lg">
                    <div className="flex-1 w-full">
                        <label className="text-sm font-medium">Enter Annual Salary (PKR)</label>
                        <Input type="number" placeholder="e.g., 1500000" value={annualSalary} onChange={e => setAnnualSalary(e.target.value)} />
                    </div>
                    <Button onClick={handleCalculateTax}>
                        <Calculator className="mr-2 h-4 w-4" /> Calculate Tax
                    </Button>
                </div>
                 {taxResult && (
                    <Card className="bg-muted/50 animate-in fade-in-50">
                        <CardHeader>
                            <CardTitle>Calculation Result</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-2">
                             <p><strong>Applicable Slab:</strong> {taxResult.slab.minIncome.toLocaleString()} - {taxResult.slab.maxIncome ? taxResult.slab.maxIncome.toLocaleString() : 'Above'}</p>
                             <p><strong>Total Annual Tax:</strong> <span className="font-bold text-primary">PKR {taxResult.totalTax.toLocaleString(undefined, { maximumFractionDigits: 2 })}</span></p>
                             <p><strong>Monthly Tax Deduction:</strong> <span className="font-bold">PKR {(taxResult.totalTax / 12).toLocaleString(undefined, { maximumFractionDigits: 2 })}</span></p>
                        </CardContent>
                    </Card>
                )}
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
