
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel, Rider, Branch } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { DatePicker } from '@/components/ui/date-picker';
import { Download, Banknote, Truck, Hash } from 'lucide-react';
import { mockParcels, mockRiders, mockBranches } from '@/lib/data';
import { format, isSameDay } from 'date-fns';
import { Badge } from '@/components/ui/badge';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';

type DailyCodTransaction = {
    parcel: Parcel;
    rider: Rider;
    branch: Branch;
};

export default function FinanceDailyCodPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [selectedDate, setSelectedDate] = useState<Date | undefined>(new Date());

  const dailyReportData = useMemo(() => {
    if (!selectedDate) return { transactions: [], totalCod: 0, totalShipments: 0 };
    
    const deliveredParcels = mockParcels.filter(p => 
        p.status === 'Delivered' && isSameDay(new Date(p.bookingDate), selectedDate)
    );

    const transactions: DailyCodTransaction[] = deliveredParcels.map((parcel, index) => ({
        parcel,
        rider: mockRiders[index % mockRiders.length],
        branch: mockBranches[index % mockBranches.length],
    }));

    const totalCod = transactions.reduce((sum, tx) => sum + tx.parcel.codAmount, 0);

    return {
        transactions,
        totalCod,
        totalShipments: transactions.length,
    }
  }, [selectedDate]);

  const averageCod = dailyReportData.totalShipments > 0 ? dailyReportData.totalCod / dailyReportData.totalShipments : 0;

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                  <div>
                    <CardTitle>Daily COD Report</CardTitle>
                    <CardDescription>
                      View a daily report of all Cash on Delivery transactions.
                    </CardDescription>
                  </div>
                  <div className="flex items-center gap-4">
                      <DatePicker date={selectedDate} setDate={setSelectedDate} />
                      <Button variant="outline" disabled={dailyReportData.transactions.length === 0}>
                        <Download className="mr-2 h-4 w-4" />
                        Export Report
                      </Button>
                  </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-8">
                <div className="grid gap-4 md:grid-cols-3">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total COD Collected</CardTitle>
                            <Banknote className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {dailyReportData.totalCod.toLocaleString()}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Shipments</CardTitle>
                            <Truck className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{dailyReportData.totalShipments}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Average COD Value</CardTitle>
                            <Hash className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">PKR {averageCod.toLocaleString(undefined, { maximumFractionDigits: 0 })}</div>
                        </CardContent>
                    </Card>
                </div>

                <div>
                    <h3 className="text-lg font-semibold mb-4">Transactions for {selectedDate ? format(selectedDate, 'PPP') : ''}</h3>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Parcel ID</TableHead>
                                <TableHead>Branch</TableHead>
                                <TableHead>Rider</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead className="text-right">COD Amount (PKR)</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {dailyReportData.transactions.length === 0 ? (
                                <TableRow>
                                    <TableCell colSpan={5} className="text-center h-24">No COD transactions for this date.</TableCell>
                                </TableRow>
                            ) : dailyReportData.transactions.map(tx => (
                                <TableRow key={tx.parcel.id}>
                                    <TableCell className="font-medium">{tx.parcel.id}</TableCell>
                                    <TableCell>{tx.branch.name}</TableCell>
                                    <TableCell>
                                        <div className="flex items-center gap-2">
                                            <Avatar className="h-8 w-8">
                                                <AvatarImage src={tx.rider.avatar} />
                                                <AvatarFallback>{tx.rider.name.charAt(0)}</AvatarFallback>
                                            </Avatar>
                                            {tx.rider.name}
                                        </div>
                                    </TableCell>
                                    <TableCell>
                                        <Badge variant="secondary" className="bg-green-100 text-green-800">{tx.parcel.status}</Badge>
                                    </TableCell>
                                    <TableCell className="text-right font-medium">{tx.parcel.codAmount.toLocaleString()}</TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                         <CardFooter>
                           <p className="text-xs text-muted-foreground mt-4">
                                This report includes all shipments marked as "Delivered" with a valid COD amount on the selected date.
                            </p>
                        </CardFooter>
                    </Table>
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
